#ifndef SCENE_HXX
#define SCENE_HXX

#include "Primitive.hxx"
#include "PerspectiveCamera.hxx"

#include "Triangle.hxx"

#include "Shader.hxx"

using namespace std;

#include "BVH.h"

class Scene
{
public:
	Vec3f bgColor; // background color

	vector<Primitive *> primitive; // primitives

	Camera *camera; 

	Shader* shd;

	BVH* bvhTree;
	Box boundingBox;

	bool m_useBVH;

	Scene(bool aUseBVH)
		: bgColor(Vec3f(0,0,0)),
		camera(new PerspectiveCamera(Vec3f(0, 0.04f, -0.12f),Vec3f(0,0,1),
		Vec3f(0,1,0),60,640,480))
	{
		shd = NULL;
		m_useBVH = aUseBVH;
	};

	virtual ~Scene() 
	{
		delete shd;
	};

	void ParseOBJ(char *fileName);

	// add a new primitive
	void Add(Primitive *prim)
	{ primitive.push_back(prim); };

	// intersect all contained objects
	virtual bool Intersect(Ray &ray)
	{
		if(m_useBVH)
		{
			return bvhTree->Intersect(ray);
		}
		else
		{
			bool hit = false;

			for (unsigned int i=0;i<primitive.size();i++)
				hit |= primitive[i]->Intersect(ray);

			return hit;
		}
	};

	// find occluder
	bool Occluded(Ray &ray)
	{
		if(m_useBVH)
		{
			return bvhTree->Intersect(ray);
		}
		else
		{
			for (unsigned int i=0;i<primitive.size();i++)
				if (primitive[i]->Occluded(ray))
					return true;

			return false;
		}
	};

	virtual Vec3f GetNormal(Ray &ray)
	{
		std::cerr << "Scene::GetNormal is not defined!" << std::endl;
		return Vec3f(0,0,0); // dummy !
	};

	Box CalcBounds()
	{
		boundingBox.Clear();

		for (int i=0;i<(int)primitive.size();i++) 
		{
			Box box = primitive[i]->CalcBounds();
			boundingBox.Extend(box);
		}

		return boundingBox;
	}

	void BuildAccelStructure()
	{
		if(m_useBVH)
		{
			boundingBox = CalcBounds();
			bvhTree = new BVH(boundingBox, primitive);

			cout << "Bounds are : " << boundingBox.min << " " << boundingBox.max << endl;
		}
	}

	// trace the given ray and shade it and
	// return the color of the shaded ray
	Vec3f RayTrace(Ray &ray)
	{
		static int maxReflections = 3;

		Vec3f result;

		// only shoot another ray if the maximum recursion depth is not exceeded
		if (maxReflections>=0 && Intersect(ray)) {
			maxReflections--; 
			result = ray.hit->shader->Shade(ray); // shade the primitive
			maxReflections++; 

			return result;
		}
		else
			return bgColor; // ray missed geometric primitives
	};
};

#endif
