package ISSearch;

import java.net.*;
import java.sql.*;
import oracle.jdbc.*;
import oracle.sql.*;

/**
 * Interface of the database connection.
 * This class is used to open and close the database connection, to prepare
 * the DB schema (tables and indexes) for storage, and to store crawled data into this schema.
 * @see Connection
 * @see ResultSet
 * @see Statement
 * @see PreparedStatement
 * @see OracleConnection
 * @see DriverManager
 */
public interface ISDBinterface
{
   /**
    * Opens the Database connection to the Oracle instance specified by input parameters.
    * @param user the Oracle user name
    * @param password the database password
    * @param hostname the hostname (or IP address) of the Oracle server
    * @param port the port of the Oracle listener for connections over a TCP/IP network (standard is 1521)
    * @param service_name the name of the requested Oracle service 
    * @return true, if the connect was succesful; false otherwise.
    */
   public boolean openConnection(String user, String password, String hostname, int port, String service_name);
   
    /**
    * Closes the open database connection.
    */
   public void closeConnection();
   
   /** Returns the current state of the database connection
    * @return true, if the database was successfully opened and is currently not yet closed.
    */
   public boolean isOpen();
   
   /**
    * Returns the internal database connection of the database interface (only when it is open and valid). 
    * This method must return the current internal database connection (when available),  
    * it should NOT open any additional connections.
    * @return the database connection (when open and valid); null otherwise.
    */
   public Connection getConnection();
   
   /** Stores the content of an ISDocument and its URL into the database. At least stems of all words
    * from the ISDocument and the absolute URL of the source Web document are required for the search engine; the storage of other available
    * informations (original words and positions, links) is optional.
    * @return true, if the storage attempt was successful; false otherwise.
    */
   public boolean store(URL url, ISDocumentInterface doc);

   /** This method creates the database infrastructure that is required for storage of crawled information: relations (tables), 
    * index structures, and integrity constraints. At least stems of all words
    * from the ISDocument and the absolute URL of the source Web document are required for the search engine. Additionally, you should provide
    * data structures for document features (stem, RTF weight, TF*IDF weight) that will be computed from stored word stems after the crawl.
    * Of course, the schema must also contain information about associations between features, terms, and URLs (e.g. using unique Doc-IDs) to   
    * allow the keyword-based search on stored data.
    * @return true, if the schema was created successfully; false otherwise.
    */
   public boolean createSchema();
   
   /**
    * This method drops all previously created schema elements (tables, associated indexes, and integrity contraints) to fully cleanup the database account 
    * for next experiment.
    */
   public void dropSchema();
   
   /** This method creates document features from previously stored word stems and stores them (as well as associated feature weights) 
    * into additional schema tables. At least, the string of the word stem, the RTF weight, and the TF*IDF weight must be stored for each 
    * feature and document. Of course, the schema must also contain information about associations between features, terms, and URLs (e.g. using unique Doc-IDs) 
    * to allow keyword-based search on stored data.
    */
   public boolean createFeatures();
}