package ISSearch;

import java.rmi.*;
import java.rmi.server.*;
import java.net.*;

/**
 * This class provides some useful functions for RMI calls
 * of the distributed Peer-to-Peer ISSEarch search engine.
 */
public final class ISPeerTools
{
   /**
    * Returns the reference to the RMI Server (Peer) of the distributed search engine.
    * The connection to the server is specified by three parameters: the hostname, the port, and the service name.
    * When the new peer server starts, it submits these parameters to the central database of the ISSearch framework.
    * The parameters are used by the client to establish the connection to the chosen server.
    * @param host The hostname of the RMI server.
    * @param port The port of the RMI service (default 1099).
    * @param serviceName The service name of the ISSerach search engine on this RMI server.
    * @exception RemoteException if the connection to the remote service cannot be established or fails.
    */
   public static ISPeerServerInterface getServer(String host, int port, String serviceName) throws Exception
   {
      String link = "//" + host + ":" + port + "/" + serviceName;
      ISPeerServerInterface result = (ISPeerServerInterface) Naming.lookup(link);
      return result;
   }
   
   /**
    * Invokes the remote search method of the RMI server.
    * @param query The query that is submitted by the client. The query syntax is prescribed by Assignment 4 (Query Processor).
    * @param numResults The max number of matches that can be returned as search results for the query.
    * @return The array of search results that contains up to <code>numResults</code> matches, sorted
    * in descending order of similarity.
    * @exception RemoteException or NotBoundException if the remote invocation of the search fails or the connection was not established.
    */
   public static ISPeerResultInterface[] search(ISPeerServerInterface server, String query, int numResults) throws Exception
   {
      return server.search(query, numResults);
   }
   
   /**
    * Converts the IP address represented by 4-byte array to the host ID of the ISSearch engine.
    * The host IDs are used to identify remote ISSearch servers.
    * @param The IP address, represented as 4-byte array.
    * @return The numerical host ID.
    */
   public static long IP2ID(byte[] ipAddr)
   {
      if (ipAddr.length != 4)
         return 0;
      long r1 = (long)(ipAddr[0] & 0xff ) << 24;
      long r2 = (long)(ipAddr[1] & 0xff ) << 16;
      long r3 = (long)(ipAddr[2] & 0xff ) << 8;
      long r4 = (long)(ipAddr[3] & 0xff ) << 0;
      long result = r1+r2+r3+r4;
      return result;
   }
   
   /**
    * Converts the numerical ID of the host into its IP address.
    * @param The host ID.
    * @return The IP address that corresponds to this ID, as 4-byte array.
    */
   public static byte[] ID2IP(long id)
   {
      byte[] byteArray = new byte[4];
      byteArray[0] = (byte)((id & 0xFF000000) >> 24);
      byteArray[1] = (byte)((id & 0xFF0000) >> 16) ;
      byteArray[2] = (byte)((id & 0xFF00) >> 8);
      byteArray[3] = (byte)((id & 0xFF) >> 0);
      
      return byteArray;
   }
   
   /**
    * Converts the array representation of the IP address (array of 4 bytes)
    * into text string.
    * @param ip The specified IP address as an 4-byte array.
    * @return The textual representation of this IP address.
    * @exception UnknownHostException, if the conversion fails.
    */
   public static String IP2String(byte[] ip) throws java.net.UnknownHostException
   {
      if (ip.length != 4)
         return null;
      return InetAddress.getByAddress(ip).getHostAddress();
   }
   
   /**
    * Returns the IP address of the current machine.
    * @return The IP address of the current machine as an array of 4 bytes.
    * @exception UnknownHostException, if the machine has no default network connection.
    */
   public static byte[] getLocalIP() throws java.net.UnknownHostException
   {
      return InetAddress.getLocalHost().getAddress();
   }
 }
