package ISSearch;

import java.io.*;
import java.net.*;
import java.util.*;
import java.lang.*;

/**
 * Interface of the main Crawler class of the Web search engine.
 * This class is used to start and stop the Crawler,
 * to reset the engine and to control crawling parameters.
 * @see Runnable
 * @see Thread
 * @see InetAddress
 * @see URL
 * @see HttpURLConnection
 * @see InputStreamReader
 * @see BufferedReader
 * @see Exception
 */
public interface ISCrawlerInterface extends Runnable
{
   
   /**
    * The Running state of the current thread
    */
   public static final int RUNNING = 100;
   
   /**
    * The Idle state of the current thread
    */
   public static final int STOPPED = 200;
   
   /**
    * Starts the thread of the crawler and changes the engine state to <code>RUNNING</code>
    */
   public void start();
   
   /**
    * Stops the crawler.
    * This method stops crawling and sets the engine status to <code>STOPPED</code>.
    */
   public void stop();
   
   /**
    * Resets the crawler.
    * This method stops the crawling,
    * resets the URL queue, and the list of visited links. 
    * Finally, it sets the crawler status to <code>STOPPED</code>, 
    */
   public void reset();
   
   /**
    * Adds a new link to the URL queue, if the link is not yet visited.
    * @param link The URL link representation of the new target
    */
   public void addLink(URL link);
   
   /**
    * Returns the current state of the crawler.
    * Possible states are <code>RUNNING</code> and <code>STOPPED</code>.
    * @return The current state of the crawler, <code>RUNNING</code> oder <code>STOPPED</code>
    */
   public int getState();
   
   /** Returns the current size of the URL queue
    * @return The current size of the URL queue.
    */
   public int getQueueSize();
   
   /** Set the maximum allowed size of the URL queue
    * @param m The maximum allowed Queue size
    */
   public void setQueueMaxSize(int m);
   
   /** Returns the maximum allowed size of the URL Queue
    * @return The max allowed Queue size
    */
   public int getMaxQueueSize();
   
   /** Sets the maximum allowed crawling depth.
    * @param depth The maximum allowed craling depth.
    */
   public void setCrawlingDepth(int depth);
   
   /** Returns the current maximum allowed crawling depth.
    * @return The current allowed craling depth.
    */
   public int getCrawlingDepth();
   
   /** Returns the best candidate to be visited next.
    *  The result must have the highest priority
    * (in the sense of the selected ordering strategy) under
    * all available links.
    * @return The best target to be visited by the Crawler next, <code>null</code> if the queue is empty.
    */
   public URL getBest();
   
   /**
    * Checks if the URL of the given document is already visited by the crawler.
    * @return <code>true</code> if the engine was able to recognize
    * the given URL as already visited, <code>false</code>.
    */
   public boolean isVisited(URL doc);
   
   /**
    * Returns the last document visited by the Crawler.
    * @return The last visited document as object that implements ISDocumentInterface (and contains all
    * extracted links, words and their stems); null if no documents were crawled yet.
    */
   public ISDocumentInterface getCurrentDocument();
   
   /**
    * Returns the last URL visited by the Crawler.
    * @return The last visited URL; null if no links were crawled yet.
    */
   public URL getCurrentURL();
}