/*******************************************************************************
 * Copyright (c) 2003 IBM Corporation and others. All rights reserved. This
 * program and the accompanying materials are made available under the terms of
 * the Common Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.ui.actions;

import org.eclipse.jface.action.IAction;

import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.help.WorkbenchHelp;

import org.eclipse.ui.internal.ActivateEditorAction;
import org.eclipse.ui.internal.CloseAllAction;
import org.eclipse.ui.internal.CloseAllPerspectivesAction;
import org.eclipse.ui.internal.CloseAllSavedAction;
import org.eclipse.ui.internal.CloseEditorAction;
import org.eclipse.ui.internal.ClosePerspectiveAction;
import org.eclipse.ui.internal.CycleEditorAction;
import org.eclipse.ui.internal.CyclePartAction;
import org.eclipse.ui.internal.CyclePerspectiveAction;
import org.eclipse.ui.internal.EditActionSetsAction;
import org.eclipse.ui.internal.IHelpContextIds;
import org.eclipse.ui.internal.IWorkbenchGraphicConstants;
import org.eclipse.ui.internal.LockToolBarAction;
import org.eclipse.ui.internal.MaximizePartAction;
import org.eclipse.ui.internal.NavigationHistoryAction;
import org.eclipse.ui.internal.OpenPreferencesAction;
import org.eclipse.ui.internal.QuitAction;
import org.eclipse.ui.internal.ResetPerspectiveAction;
import org.eclipse.ui.internal.SaveAction;
import org.eclipse.ui.internal.SaveAllAction;
import org.eclipse.ui.internal.SaveAsAction;
import org.eclipse.ui.internal.SavePerspectiveAction;
import org.eclipse.ui.internal.ShowPartPaneMenuAction;
import org.eclipse.ui.internal.ShowViewMenuAction;
import org.eclipse.ui.internal.ToggleEditorsVisibilityAction;
import org.eclipse.ui.internal.WorkbenchEditorsAction;
import org.eclipse.ui.internal.WorkbenchImages;
import org.eclipse.ui.internal.WorkbenchMessages;
import org.eclipse.ui.internal.WorkbenchWindow;
import org.eclipse.ui.internal.actions.ActivityEnablerAction;

/**
 * Access to standard actions provided by the workbench.
 * <p>
 * Most of the functionality of this class is provided by static methods and
 * fields. Example usage:
 * 
 * <pre>
 * MenuManager menu = ...; 
 * ActionFactory.IWorkbenchAction closeEditorAction 
 *    = ActionFactory.CLOSE.create(window); 
 * menu.add(closeEditorAction);
 * </pre>
 * </p>
 * <p>
 * Clients may declare other classes that provide additional application-specific
 * action factories.
 * </p>
 * 
 * @since 3.0
 */
public abstract class ActionFactory {

	/**
	 * Interface for a workbench action.
	 * <p>
	 * This interface is not intended to be implemented by clients.
	 * </p>
	 */
	public interface IWorkbenchAction extends IAction {
		/**
		 * Disposes of this action. Once disposed, this action cannot be used.
		 * This operation has no effect if the action has already been
		 * disposed.
		 */
		public void dispose();
	}

	/**
	 * Workbench action (id "activateEditor"): Activate the most recently used
	 * editor. This action maintains its enablement state.
	 */
		public static final ActionFactory ACTIVATE_EDITOR = new ActionFactory("activateEditor") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new ActivateEditorAction(window);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "back"): Back. This action is a
	 * {@link Retarget Retarget}action with id "back". This action maintains
	 * its enablement state.
	 */
		public static final ActionFactory BACK = new ActionFactory("back") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			RetargetAction action = new LabelRetargetAction(getId(), WorkbenchMessages.getString("Workbench.back")); //$NON-NLS-1$  //$NON-NLS-2$
			action.setToolTipText(WorkbenchMessages.getString("Workbench.backToolTip")); //$NON-NLS-1$
			window.getPartService().addPartListener(action);
			action.setActionDefinitionId("org.eclipse.ui.navigate.back"); //$NON-NLS-1$
			return action;
		}
	};

	/**
	 * Workbench action (id "backardHistory"): Backward in the navigation
	 * history. This action maintains its enablement state.
	 */
		public static final ActionFactory BACKWARD_HISTORY = new ActionFactory("backardHistory") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new NavigationHistoryAction(window, false);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "close"): Close the active editor. This action
	 * maintains its enablement state.
	 */
		public static final ActionFactory CLOSE = new ActionFactory("close") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new CloseEditorAction(window);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "closeAll"): Close all open editors. This action
	 * maintains its enablement state.
	 */
		public static final ActionFactory CLOSE_ALL = new ActionFactory("closeAll") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new CloseAllAction(window);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "closeAllPerspectives"): Closes all perspectives.
	 * This action maintains its enablement state.
	 */
		public static final ActionFactory CLOSE_ALL_PERSPECTIVES = new ActionFactory("closeAllPerspectives") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new CloseAllPerspectivesAction(window);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "closeAllSaved"): Close all open editors except
	 * those with unsaved changes. This action maintains its enablement state.
	 */
		public static final ActionFactory CLOSE_ALL_SAVED = new ActionFactory("closeAllSaved") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new CloseAllSavedAction(window);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "closePerspective"): Closes the current
	 * perspective. This action maintains its enablement state.
	 */
		public static final ActionFactory CLOSE_PERSPECTIVE = new ActionFactory("closePerspective") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new ClosePerspectiveAction(window);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "configureActivities"): Activity Configuration This
	 * action launches the dialog which allows for post-install configuration
	 * of enabled activities. This action should be configurable via extension
	 * point in some fashion with the action shown here as the default.
	 */
		public static final ActionFactory CONFIGURE_ACTIVITIES = new ActionFactory("configureActivities") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new ActivityEnablerAction(window);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "copy"): Copy. This action is a
	 * {@link Retarget Retarget}action with id "copy". This action maintains
	 * its enablement state.
	 */
		public static final ActionFactory COPY = new ActionFactory("copy") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			RetargetAction action = new RetargetAction(getId(), WorkbenchMessages.getString("Workbench.copy")); //$NON-NLS-1$  //$NON-NLS-2$
			action.setToolTipText(WorkbenchMessages.getString("Workbench.copyToolTip")); //$NON-NLS-1$
			window.getPartService().addPartListener(action);
			action.setActionDefinitionId("org.eclipse.ui.edit.copy"); //$NON-NLS-1$
			ISharedImages sharedImages = window.getWorkbench().getSharedImages();
			action.setImageDescriptor(sharedImages.getImageDescriptor(ISharedImages.IMG_TOOL_COPY));
			action.setHoverImageDescriptor(
				sharedImages.getImageDescriptor(ISharedImages.IMG_TOOL_COPY_HOVER));
			action.setDisabledImageDescriptor(
				sharedImages.getImageDescriptor(ISharedImages.IMG_TOOL_COPY_DISABLED));
			return action;
		}
	};

	/**
	 * Workbench action (id "cut"): Cut. This action is a
	 * {@link Retarget Retarget}action with id "cut". This action maintains
	 * its enablement state.
	 */
		public static final ActionFactory CUT = new ActionFactory("cut") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			RetargetAction action = new RetargetAction(getId(), WorkbenchMessages.getString("Workbench.cut")); //$NON-NLS-1$ //$NON-NLS-2$
			action.setToolTipText(WorkbenchMessages.getString("Workbench.cutToolTip")); //$NON-NLS-1$
			window.getPartService().addPartListener(action);
			action.setActionDefinitionId("org.eclipse.ui.edit.cut"); //$NON-NLS-1$
			ISharedImages sharedImages = window.getWorkbench().getSharedImages();
			action.setImageDescriptor(sharedImages.getImageDescriptor(ISharedImages.IMG_TOOL_CUT));
			action.setHoverImageDescriptor(
				sharedImages.getImageDescriptor(ISharedImages.IMG_TOOL_CUT_HOVER));
			action.setDisabledImageDescriptor(
				sharedImages.getImageDescriptor(ISharedImages.IMG_TOOL_CUT_DISABLED));
			return action;
		}
	};

	/**
	 * Workbench action (id "delete"): Delete. This action is a
	 * {@link Retarget Retarget}action with id "delete". This action maintains
	 * its enablement state.
	 */
		public static final ActionFactory DELETE = new ActionFactory("delete") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			RetargetAction action = new RetargetAction(getId(), WorkbenchMessages.getString("Workbench.delete")); //$NON-NLS-1$  //$NON-NLS-2$
			action.setToolTipText(WorkbenchMessages.getString("Workbench.deleteToolTip")); //$NON-NLS-1$
			window.getPartService().addPartListener(action);
			action.setActionDefinitionId("org.eclipse.ui.edit.delete"); //$NON-NLS-1$
			action.enableAccelerator(false);
			WorkbenchHelp.setHelp(action, IHelpContextIds.DELETE_RETARGET_ACTION);
			ISharedImages sharedImages = window.getWorkbench().getSharedImages();
			action.setImageDescriptor(
				sharedImages.getImageDescriptor(ISharedImages.IMG_TOOL_DELETE));
			action.setHoverImageDescriptor(
				sharedImages.getImageDescriptor(ISharedImages.IMG_TOOL_DELETE_HOVER));
			action.setDisabledImageDescriptor(
				sharedImages.getImageDescriptor(ISharedImages.IMG_TOOL_DELETE_DISABLED));
			return action;
		}
	};

	/**
	 * Workbench action (id "editActionSets"): Edit the action sets. This
	 * action maintains its enablement state.
	 */
		public static final ActionFactory EDIT_ACTION_SETS = new ActionFactory("editActionSets") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new EditActionSetsAction(window);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "export"): Opens the export wizard. This action
	 * maintains its enablement state.
	 */
		public static final ActionFactory EXPORT = new ActionFactory("export") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new ExportResourcesAction(window);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "find"): Find. This action is a
	 * {@link Retarget Retarget}action with id "find". This action maintains
	 * its enablement state.
	 */
		public static final ActionFactory FIND = new ActionFactory("find") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			RetargetAction action = new RetargetAction(getId(), WorkbenchMessages.getString("Workbench.findReplace")); //$NON-NLS-1$  //$NON-NLS-2$
			action.setToolTipText(WorkbenchMessages.getString("Workbench.findReplaceToolTip")); //$NON-NLS-1$
			window.getPartService().addPartListener(action);
			action.setActionDefinitionId("org.eclipse.ui.edit.findReplace"); //$NON-NLS-1$
			// Find's images are commented out due to a conflict with Search.
			// See bug 16412.
			//		action.setImageDescriptor(WorkbenchImages.getImageDescriptor(IWorkbenchGraphicConstants.IMG_CTOOL_SEARCH_SRC));
			//		action.setHoverImageDescriptor(WorkbenchImages.getImageDescriptor(IWorkbenchGraphicConstants.IMG_CTOOL_SEARCH_SRC_HOVER));
			//		action.setDisabledImageDescriptor(WorkbenchImages.getImageDescriptor(IWorkbenchGraphicConstants.IMG_CTOOL_SEARCH_SRC_DISABLED));
			return action;
		}
	};

	/**
	 * Workbench action (id "forward"): Forward. This action is a
	 * {@link Retarget Retarget}action with id "forward". This action
	 * maintains its enablement state.
	 */
		public static final ActionFactory FORWARD = new ActionFactory("forward") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			RetargetAction action = new LabelRetargetAction(getId(), WorkbenchMessages.getString("Workbench.forward")); //$NON-NLS-1$  //$NON-NLS-2$
			action.setToolTipText(WorkbenchMessages.getString("Workbench.forwardToolTip")); //$NON-NLS-1$
			window.getPartService().addPartListener(action);
			action.setActionDefinitionId("org.eclipse.ui.navigate.forward"); //$NON-NLS-1$
			return action;
		}
	};

	/**
	 * Workbench action (id "forwardHistory"): Forward in the navigation
	 * history. This action maintains its enablement state.
	 */
		public static final ActionFactory FORWARD_HISTORY = new ActionFactory("forwardHistory") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new NavigationHistoryAction(window, true);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "goInto"): Go Into. This action is a
	 * {@link Retarget Retarget}action with id "goInto". This action maintains
	 * its enablement state.
	 */
		public static final ActionFactory GO_INTO = new ActionFactory("goInto") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			RetargetAction action = new LabelRetargetAction(getId(), WorkbenchMessages.getString("Workbench.goInto")); //$NON-NLS-1$  //$NON-NLS-2$
			action.setToolTipText(WorkbenchMessages.getString("Workbench.goIntoToolTip")); //$NON-NLS-1$
			window.getPartService().addPartListener(action);
			action.setActionDefinitionId("org.eclipse.ui.navigate.goInto"); //$NON-NLS-1$
			return action;
		}
	};

	/**
	 * Workbench action (id "import"): Opens the import wizard. This action
	 * maintains its enablement state.
	 */
		public static final ActionFactory IMPORT = new ActionFactory("import") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new ImportResourcesAction(window);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "lockToolBar"): Lock/unlock the workbench window
	 * tool bar. This action maintains its enablement state.
	 */
		public static final ActionFactory LOCK_TOOL_BAR = new ActionFactory("lockToolBar") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new LockToolBarAction(window);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "maximize"): Maximize/restore the active part. This
	 * action maintains its enablement state.
	 */
		public static final ActionFactory MAXIMIZE = new ActionFactory("maximize") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new MaximizePartAction(window);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "move"): Move. This action is a
	 * {@link Retarget Retarget}action with id "move". This action maintains
	 * its enablement state.
	 */
		public static final ActionFactory MOVE = new ActionFactory("move") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			RetargetAction action = new RetargetAction(getId(), WorkbenchMessages.getString("Workbench.move")); //$NON-NLS-1$  //$NON-NLS-2$
			action.setToolTipText(WorkbenchMessages.getString("Workbench.moveToolTip")); //$NON-NLS-1$
			window.getPartService().addPartListener(action);
			action.setActionDefinitionId("org.eclipse.ui.edit.move"); //$NON-NLS-1$
			return action;
		}
	};

	/**
	 * Workbench action (id "new"): Opens the new wizard. This action maintains
	 * its enablement state.
	 */
		public static final ActionFactory NEW = new ActionFactory("new") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new NewWizardAction(window);
			action.setId(getId());
			// indicate that a new wizard submenu has been created
			 ((WorkbenchWindow) window).addSubmenu(WorkbenchWindow.NEW_WIZARD_SUBMENU);
			return action;
		}
	};

	/**
	 * Workbench action (id "next"): Next. This action is a
	 * {@link Retarget Retarget}action with id "next". This action maintains
	 * its enablement state.
	 */
		public static final ActionFactory NEXT = new ActionFactory("next") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			RetargetAction action = new LabelRetargetAction(getId(), WorkbenchMessages.getString("Workbench.next")); //$NON-NLS-1$  //$NON-NLS-2$
			action.setToolTipText(WorkbenchMessages.getString("Workbench.nextToolTip")); //$NON-NLS-1$
			window.getPartService().addPartListener(action);
			action.setActionDefinitionId("org.eclipse.ui.navigate.next"); //$NON-NLS-1$
			return action;
		}
	};

	/**
	 * Workbench action (id "nextEditor"): Next editor. This action maintains
	 * its enablement state.
	 * <p>
	 * <code>NEXT_EDITOR</code> and <code>PREVIOUS_EDITOR</code> form a
	 * cycle action pair. For a given window, use
	 * {@link ActionFactory#linkCycleActionPair
	 * ActionFactory.linkCycleActionPair</code>} to connect the two.
	 * </p>
	 */
		public static final ActionFactory NEXT_EDITOR = new ActionFactory("nextEditor") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new CycleEditorAction(window, true);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "nextPart"): Next part. This action maintains its
	 * enablement state.
	 * <p>
	 * <code>NEXT_PART</code> and <code>PREVIOUS_PART</code> form a cycle
	 * action pair. For a given window, use
	 * {@link ActionFactory#linkCycleActionPair
	 * ActionFactory.linkCycleActionPair</code>} to connect the two.
	 * </p>
	 */
		public static final ActionFactory NEXT_PART = new ActionFactory("nextPart") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new CyclePartAction(window, true);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "nextPerspective"): Next perspective. This action
	 * maintains its enablement state.
	 * <p>
	 * <code>NEXT_PERSPECTIVE</code> and <code>PREVIOUS_PERSPECTIVE</code>
	 * form a cycle action pair. For a given window, use
	 * {@link ActionFactory#linkCycleActionPair
	 * ActionFactory.linkCycleActionPair</code>} to connect the two.
	 * </p>
	 */
		public static final ActionFactory NEXT_PERSPECTIVE = new ActionFactory("nextPerspective") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new CyclePerspectiveAction(window, true);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "openNewWindow"): Open a new workbench window. This
	 * action maintains its enablement state.
	 */
		public static final ActionFactory OPEN_NEW_WINDOW = new ActionFactory("openNewWindow") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new OpenInNewWindowAction(window);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "paste"): Paste. This action is a
	 * {@link Retarget Retarget}action with id "paste". This action maintains
	 * its enablement state.
	 */
		public static final ActionFactory PASTE = new ActionFactory("paste") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			RetargetAction action = new RetargetAction(getId(), WorkbenchMessages.getString("Workbench.paste")); //$NON-NLS-1$  //$NON-NLS-2$
			action.setToolTipText(WorkbenchMessages.getString("Workbench.pasteToolTip")); //$NON-NLS-1$
			window.getPartService().addPartListener(action);
			action.setActionDefinitionId("org.eclipse.ui.edit.paste"); //$NON-NLS-1$
			ISharedImages sharedImages = window.getWorkbench().getSharedImages();
			action.setImageDescriptor(
				sharedImages.getImageDescriptor(ISharedImages.IMG_TOOL_PASTE));
			action.setHoverImageDescriptor(
				sharedImages.getImageDescriptor(ISharedImages.IMG_TOOL_PASTE_HOVER));
			action.setDisabledImageDescriptor(
				sharedImages.getImageDescriptor(ISharedImages.IMG_TOOL_PASTE_DISABLED));
			return action;
		}
	};

	/**
	 * Workbench action (id "preferences"): Displays the Preferences dialog.
	 * This action maintains its enablement state.
	 */
		public static final ActionFactory PREFERENCES = new ActionFactory("preferences") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new OpenPreferencesAction(window);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "previous"): Previous. This action is a
	 * {@link Retarget Retarget}action with id "previous". This action
	 * maintains its enablement state.
	 */
		public static final ActionFactory PREVIOUS = new ActionFactory("previous") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			RetargetAction action = new LabelRetargetAction(getId(), WorkbenchMessages.getString("Workbench.previous")); //$NON-NLS-1$  //$NON-NLS-2$
			action.setToolTipText(WorkbenchMessages.getString("Workbench.previousToolTip")); //$NON-NLS-1$
			window.getPartService().addPartListener(action);
			action.setActionDefinitionId("org.eclipse.ui.navigate.previous"); //$NON-NLS-1$
			return action;
		}
	};

	/**
	 * Workbench action (id "previousEditor"): Previous editor. This action
	 * maintains its enablement state.
	 * <p>
	 * <code>NEXT_EDITOR</code> and <code>PREVIOUS_EDITOR</code> form a
	 * cycle action pair. For a given window, use
	 * {@link ActionFactory#linkCycleActionPair
	 * ActionFactory.linkCycleActionPair</code>} to connect the two.
	 * </p>
	 */
		public static final ActionFactory PREVIOUS_EDITOR = new ActionFactory("previousEditor") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new CycleEditorAction(window, false);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "previousPart"): Previous part. This action
	 * maintains its enablement state.
	 * <p>
	 * <code>NEXT_PART</code> and <code>PREVIOUS_PART</code> form a cycle
	 * action pair. For a given window, use
	 * {@link ActionFactory#linkCycleActionPair
	 * ActionFactory.linkCycleActionPair</code>} to connect the two.
	 * </p>
	 */
		public static final ActionFactory PREVIOUS_PART = new ActionFactory("previousPart") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new CyclePartAction(window, false);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "previousPerspective"): Previous perspective. This
	 * action maintains its enablement state.
	 * <p>
	 * <code>NEXT_PERSPECTIVE</code> and <code>PREVIOUS_PERSPECTIVE</code>
	 * form a cycle action pair. For a given window, use
	 * {@link ActionFactory#linkCycleActionPair
	 * ActionFactory.linkCycleActionPair</code>} to connect the two.
	 * </p>
	 */
		public static final ActionFactory PREVIOUS_PERSPECTIVE = new ActionFactory("previousPerspective") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new CyclePerspectiveAction(window, false);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "print"): Print. This action is a
	 * {@link Retarget Retarget}action with id "print". This action maintains
	 * its enablement state.
	 */
		public static final ActionFactory PRINT = new ActionFactory("print") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			RetargetAction action = new RetargetAction(getId(), WorkbenchMessages.getString("Workbench.print")); //$NON-NLS-1$  //$NON-NLS-2$
			action.setToolTipText(WorkbenchMessages.getString("Workbench.printToolTip")); //$NON-NLS-1$
			window.getPartService().addPartListener(action);
			action.setActionDefinitionId("org.eclipse.ui.file.print"); //$NON-NLS-1$
			action.setImageDescriptor(
				WorkbenchImages.getImageDescriptor(
					IWorkbenchGraphicConstants.IMG_CTOOL_PRINT_EDIT));
			action.setHoverImageDescriptor(
				WorkbenchImages.getImageDescriptor(
					IWorkbenchGraphicConstants.IMG_CTOOL_PRINT_EDIT_HOVER));
			action.setDisabledImageDescriptor(
				WorkbenchImages.getImageDescriptor(
					IWorkbenchGraphicConstants.IMG_CTOOL_PRINT_EDIT_DISABLED));
			return action;
		}
	};

	/**
	 * Workbench action (id "properties"): Properties. This action is a
	 * {@link Retarget Retarget}action with id "properties". This action
	 * maintains its enablement state.
	 */
		public static final ActionFactory PROPERTIES = new ActionFactory("properties") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			RetargetAction action = new RetargetAction(getId(), WorkbenchMessages.getString("Workbench.properties")); //$NON-NLS-1$  //$NON-NLS-2$
			action.setToolTipText(WorkbenchMessages.getString("Workbench.propertiesToolTip")); //$NON-NLS-1$
			window.getPartService().addPartListener(action);
			action.setActionDefinitionId("org.eclipse.ui.file.properties"); //$NON-NLS-1$
			return action;
		}
	};

	/**
	 * Workbench action (id "quit"): Quit (close the workbench). This action
	 * maintains its enablement state.
	 */
		public static final ActionFactory QUIT = new ActionFactory("quit") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new QuitAction(window);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "redo"): Redo. This action is a
	 * {@link Retarget Retarget}action with id "redo". This action maintains
	 * its enablement state.
	 */
		public static final ActionFactory REDO = new ActionFactory("redo") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			LabelRetargetAction action = new LabelRetargetAction(getId(), WorkbenchMessages.getString("Workbench.redo")); //$NON-NLS-1$  //$NON-NLS-2$
			action.setToolTipText(WorkbenchMessages.getString("Workbench.redoToolTip")); //$NON-NLS-1$
			window.getPartService().addPartListener(action);
			action.setActionDefinitionId("org.eclipse.ui.edit.redo"); //$NON-NLS-1$
			ISharedImages sharedImages = window.getWorkbench().getSharedImages();
			action.setImageDescriptor(sharedImages.getImageDescriptor(ISharedImages.IMG_TOOL_REDO));
			action.setHoverImageDescriptor(
				sharedImages.getImageDescriptor(ISharedImages.IMG_TOOL_REDO_HOVER));
			action.setDisabledImageDescriptor(
				sharedImages.getImageDescriptor(ISharedImages.IMG_TOOL_REDO_DISABLED));
			return action;
		}
	};

	/**
	 * Workbench action (id "refresh"): Refresh. This action is a
	 * {@link Retarget Retarget}action with id "refresh". This action
	 * maintains its enablement state.
	 */
		public static final ActionFactory REFRESH = new ActionFactory("refresh") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			RetargetAction action = new RetargetAction(getId(), WorkbenchMessages.getString("Workbench.refresh")); //$NON-NLS-1$  //$NON-NLS-2$
			action.setToolTipText(WorkbenchMessages.getString("Workbench.refreshToolTip")); //$NON-NLS-1$
			window.getPartService().addPartListener(action);
			action.setActionDefinitionId("org.eclipse.ui.file.refresh"); //$NON-NLS-1$
			return action;
		}
	};

	/**
	 * Workbench action (id "rename"): Rename. This action is a
	 * {@link Retarget Retarget}action with id "rename". This action maintains
	 * its enablement state.
	 */
		public static final ActionFactory RENAME = new ActionFactory("rename") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			RetargetAction action = new RetargetAction(getId(), WorkbenchMessages.getString("Workbench.rename")); //$NON-NLS-1$  //$NON-NLS-2$
			action.setToolTipText(WorkbenchMessages.getString("Workbench.renameToolTip")); //$NON-NLS-1$
			window.getPartService().addPartListener(action);
			action.setActionDefinitionId("org.eclipse.ui.edit.rename"); //$NON-NLS-1$
			return action;
		}
	};

	/**
	 * Workbench action (id "resetPerspective"): Resets the current
	 * perspective. This action maintains its enablement state.
	 */
		public static final ActionFactory RESET_PERSPECTIVE = new ActionFactory("resetPerspective") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new ResetPerspectiveAction(window);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "revert"): Revert. This action is a
	 * {@link Retarget Retarget}action with id "revert". This action maintains
	 * its enablement state.
	 */
		public static final ActionFactory REVERT = new ActionFactory("revert") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			RetargetAction action = new RetargetAction(getId(), WorkbenchMessages.getString("Workbench.revert")); //$NON-NLS-1$  //$NON-NLS-2$
			action.setToolTipText(WorkbenchMessages.getString("Workbench.revertToolTip")); //$NON-NLS-1$
			window.getPartService().addPartListener(action);
			action.setActionDefinitionId("org.eclipse.ui.file.revert"); //$NON-NLS-1$
			return action;
		}
	};

	/**
	 * Workbench action (id "save"): Save the active editor. This action
	 * maintains its enablement state.
	 */
		public static final ActionFactory SAVE = new ActionFactory("save") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new SaveAction(window);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "saveAll"): Save all open editors with unsaved
	 * changes. This action maintains its enablement state.
	 */
		public static final ActionFactory SAVE_ALL = new ActionFactory("saveAll") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new SaveAllAction(window);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "saveAs"): Save As for the active editor. This
	 * action maintains its enablement state.
	 */
		public static final ActionFactory SAVE_AS = new ActionFactory("saveAs") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new SaveAsAction(window);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "savePerspective"): Save the current perspective.
	 * This action maintains its enablement state.
	 */
		public static final ActionFactory SAVE_PERSPECTIVE = new ActionFactory("savePerspective") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new SavePerspectiveAction(window);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "selectAll"): Select All. This action is a
	 * {@link Retarget Retarget}action with id "selectAll". This action
	 * maintains its enablement state.
	 */
		public static final ActionFactory SELECT_ALL = new ActionFactory("selectAll") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			RetargetAction action = new RetargetAction(getId(), WorkbenchMessages.getString("Workbench.selectAll")); //$NON-NLS-1$  //$NON-NLS-2$
			action.setToolTipText(WorkbenchMessages.getString("Workbench.selectAllToolTip")); //$NON-NLS-1$
			window.getPartService().addPartListener(action);
			action.setActionDefinitionId("org.eclipse.ui.edit.selectAll"); //$NON-NLS-1$
			return action;
		}
	};

	/**
	 * Workbench action (id "showEditor"): Show/hide the editor area. This
	 * action maintains its enablement state.
	 */
		public static final ActionFactory SHOW_EDITOR = new ActionFactory("showEditor") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new ToggleEditorsVisibilityAction(window);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "showOpenEditors"): Show a list of open (and
	 * recently closed) editors. This action maintains its enablement state.
	 */
		public static final ActionFactory SHOW_OPEN_EDITORS = new ActionFactory("showOpenEditors") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new WorkbenchEditorsAction(window);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "showPartPaneMenu"): Show the part pane menu. This
	 * action maintains its enablement state.
	 */
		public static final ActionFactory SHOW_PART_PANE_MENU = new ActionFactory("showPartPaneMenu") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new ShowPartPaneMenuAction(window);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "showViewMenu"): Show the view menu. This action
	 * maintains its enablement state.
	 */
		public static final ActionFactory SHOW_VIEW_MENU = new ActionFactory("showViewMenu") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			IWorkbenchAction action = new ShowViewMenuAction(window);
			action.setId(getId());
			return action;
		}
	};

	/**
	 * Workbench action (id "undo"): Undo. This action is a
	 * {@link Retarget Retarget}action with id "undo". This action maintains
	 * its enablement state.
	 */
		public static final ActionFactory UNDO = new ActionFactory("undo") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			LabelRetargetAction action = new LabelRetargetAction(getId(), WorkbenchMessages.getString("Workbench.undo")); //$NON-NLS-1$  //$NON-NLS-2$
			action.setToolTipText(WorkbenchMessages.getString("Workbench.undoToolTip")); //$NON-NLS-1$
			window.getPartService().addPartListener(action);
			action.setActionDefinitionId("org.eclipse.ui.edit.undo"); //$NON-NLS-1$
			ISharedImages sharedImages = window.getWorkbench().getSharedImages();
			action.setImageDescriptor(sharedImages.getImageDescriptor(ISharedImages.IMG_TOOL_UNDO));
			action.setHoverImageDescriptor(
				sharedImages.getImageDescriptor(ISharedImages.IMG_TOOL_UNDO_HOVER));
			action.setDisabledImageDescriptor(
				sharedImages.getImageDescriptor(ISharedImages.IMG_TOOL_UNDO_DISABLED));
			return action;
		}
	};

	/**
	 * Workbench action (id "up"): Up. This action is a
	 * {@link Retarget Retarget}action with id "up". This action maintains its
	 * enablement state.
	 */
		public static final ActionFactory UP = new ActionFactory("up") {//$NON-NLS-1$
	/* (non-javadoc) method declared on ActionFactory */
		public IWorkbenchAction create(IWorkbenchWindow window) {
			if (window == null) {
				throw new IllegalArgumentException();
			}
			RetargetAction action = new LabelRetargetAction(getId(), WorkbenchMessages.getString("Workbench.up")); //$NON-NLS-1$  //$NON-NLS-2$
			action.setToolTipText(WorkbenchMessages.getString("Workbench.upToolTip")); //$NON-NLS-1$
			window.getPartService().addPartListener(action);
			action.setActionDefinitionId("org.eclipse.ui.navigate.up"); //$NON-NLS-1$
			return action;
		}
	};

	/**
	 * Establishes bi-direction connections between the forward and backward
	 * actions of a cycle pair.
	 * <p>
	 * Example usage:
	 * 
	 * <pre>
	 *  ActionFactory.IWorkbenchAction nextEditorAction = ActionFactory.NEXT_EDITOR.create(window); ActionFactory.IWorkbenchAction previousEditorAction = ActionFactory.PREVIOUS_EDITOR.create(window); ActionFactory.linkCycleActionPair(nextEditorAction, previousEditorAction);
	 * </pre>
	 * 
	 * </p>
	 * 
	 * @param next
	 *            the action that moves forward
	 * @param previous
	 *            the action that moves backward
	 */
	public static void linkCycleActionPair(IWorkbenchAction next, IWorkbenchAction previous) {
		if (!(next instanceof CyclePartAction)) {
			throw new IllegalArgumentException();
		}
		if (!(previous instanceof CyclePartAction)) {
			throw new IllegalArgumentException();
		}
		CyclePartAction n = (CyclePartAction) next;
		CyclePartAction p = (CyclePartAction) previous;
		n.setForwardActionDefinitionId(next.getActionDefinitionId());
		n.setBackwardActionDefinitionId(previous.getActionDefinitionId());
		p.setForwardActionDefinitionId(next.getActionDefinitionId());
		p.setBackwardActionDefinitionId(previous.getActionDefinitionId());
	}

	/**
	 * Id of actions created by this action factory.
	 */
	private final String actionId;

	/**
	 * Creates a new workbench action factory with the given id.
	 * 
	 * @param actionId
	 *            the id of actions created by this action factory
	 */
	protected ActionFactory(String actionId) {
		this.actionId = actionId;
	}

	/**
	 * Creates a new standard action for the given workbench window. The action
	 * has an id as specified by the particular factory.
	 * <p>
	 * Actions automatically register listeners against the workbench window so
	 * that they can keep their enablement state up to date. Ordinarily, the
	 * window's references to these listeners will be dropped automatically
	 * when the window closes. However, if the client needs to get rid of an
	 * action while the window is still open, the client must call
	 * {@link IWorkbenchAction#dispose dispose}to give the action an
	 * opportunity to deregister its listeners and to perform any other
	 * cleanup.
	 * </p>
	 * 
	 * @param window
	 *            the workbench window
	 * @return the workbench action
	 */
	public abstract IWorkbenchAction create(IWorkbenchWindow window);

	/**
	 * Returns the id of this action factory.
	 * 
	 * @return the id of actions created by this action factory
	 */
	public String getId() {
		return actionId;
	}

}
