/*******************************************************************************
 * Copyright (c) 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.osgi.framework.adaptor;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.security.ProtectionDomain;
import java.util.Dictionary;
import java.util.Enumeration;

import org.osgi.framework.Bundle;
import org.osgi.framework.BundleException;

/**
 * The <code>BundleData</code> represents a single bundle that is persistently 
 * stored by a <code>FrameworkAdaptor</code>.  A <code>BundleData</code> creates
 * the ClassLoader for a bundle, finds native libraries installed in the
 * FrameworkAdaptor for the bundle, creates data files for the bundle,
 * used to access bundle entries, manifest information, and getting and saving
 * metadata.
 */
public interface BundleData
{
	/**
	 * Creates the ClassLoader for the BundleData.  The ClassLoader created
	 * must use the <code>ClassLoaderDelegate</code> to delegate class, resource
	 * and library loading.  The delegate is responsible for finding any resource
	 * or classes imported by the bundle or provided by bundle fragments or 
	 * bundle hosts.  The <code>ProtectionDomain</code> domain must be used
	 * by the Classloader when defining a class.  
	 * @param delegate The <code>ClassLoaderDelegate</code> to delegate to.
	 * @param domain The <code>ProtectionDomain</code> to use when defining a class.
	 * @param bundleclasspath An array of bundle classpaths to use to create this
	 * classloader.  This is specified by the Bundle-ClassPath manifest entry.
	 * @return The new ClassLoader for the BundleData.
	 */
	public BundleClassLoader createClassLoader(ClassLoaderDelegate delegate, ProtectionDomain domain, String[] bundleclasspath);

	/**
	 * Gets a <code>URL</code> to the bundle entry specified by path.
	 * This method must not use the BundleClassLoader to find the
	 * bundle entry since the ClassLoader will delegate to find the resource.
	 * @param path The bundle entry path.
	 * @return A URL used to access the entry or null if the entry
	 * does not exist.
	 */
	public URL getEntry(String path);

	/**
	 * Gets all of the bundle entries that exist under the specified path.
	 * For example: <p>
	 * <code>getEntryPaths("/META-INF")</code> <p>
	 * This will return all entries from the /META-INF directory of the bundle.
	 * @param path The path to a directory in the bundle.
	 * @return An Enumeration of the entry paths or null if the specified path
	 * does not exist.
	 */
	public Enumeration getEntryPaths(String path);

	/**
	 * Returns the absolute path name of a native library. The BundleData
	 * ClassLoader invokes this method to locate the native libraries that 
	 * belong to classes loaded from this BundleData. Returns 
	 * null if the library does not exist in this BundleData.
	 * @param libname The name of the library to find the absolute path to.
	 * @return The absolute path name of the native library or null if
	 * the library does not exist.
	 */
	public String findLibrary(String libname);

	/**
	 * Installs the native code paths for this BundleData.  Each
	 * element of nativepaths must be installed for lookup when findLibrary 
	 * is called.
	 * @param nativepaths The array of native code paths to install for
	 * the bundle.
	 * @throws BundleException If any error occurs during install.
	 */
	public void installNativeCode(String[] nativepaths) throws BundleException;

	/**
	 * Return the bundle data directory.
	 * Attempt to create the directory if it does not exist.
	 *
	 * @return Bundle data directory.
	 */

	public File getDataFile(String path);

	/**
	 * Return the BundleManifest for the BundleData.
	 * @return Dictionary for containing the Manifest headers for the BundleData.
	 * @throws BundleException if an error occurred while reading the
	 * bundle manifest data.
	 */
	public Dictionary getManifest() throws BundleException;

	/**
	 * Get the BundleData bundle ID.  This will be used as the bundle
	 * ID by the framework.
	 * @return The BundleData ID.
	 */
	public long getBundleID();

	/**
	 * Get the BundleData Location.  This will be used as the bundle
	 * location by the framework.
	 * @return the BundleData location.
	 */
	public String getLocation();

	/**
	 * Return a Dictionary of all manifest headers.
	 * @return all manifest headers.
	 */
	public Dictionary getHeaders();

	/**
	 * Close all resources for this BundleData
	 * @throws IOException If an error occurs closing.
	 */
	public void close() throws IOException;

	/**
     * Open the BundleData. This method will reopen the BundleData if it has been
     * previously closed.
	 * @throws IOException If an error occurs opening.
	 */
	public void open() throws IOException;

	/**
	 * Sets the Bundle object for this BundleData.
	 * @param bundle The Bundle Object for this BundleData.
	 */
	public void setBundle(Bundle bundle);
	
	public int getStartLevel();
	public int getStatus();
	public void setStartLevel(int value);
	public void setStatus(int value);
	public void save() throws IOException;

	public String getUniqueId();
	public Version getVersion();
	public boolean isFragment();
	public String getClassPath();
	public String getActivator();
	public String getExecutionEnvironment();
	public String getDynamicImports();
}
