/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.jdt.internal.core;

import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaModelStatus;
import org.eclipse.jdt.core.IJavaModelStatusConstants;
import org.eclipse.jdt.core.ISourceReference;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.internal.core.util.Util;

/**
 * This operation renames elements.
 *
 * <p>Notes:<ul>
 * <li>Resource rename is not supported - this operation only renames
 *	   elements contained in compilation units.
 * <li>When a main type is renamed, its compilation unit and constructors are renamed.
 * <li>Constructors cannot be renamed.
 * </ul>
 */
public class RenameElementsOperation extends MoveElementsOperation {
/**
 * When executed, this operation will rename the specified elements with the given names in the
 * corresponding destinations.
 */
public RenameElementsOperation(IJavaElement[] elements, IJavaElement[] destinations, String[] newNames, boolean force) {
	//a rename is a move to the same parent with a new name specified
	//these elements are from different parents
	super(elements, destinations, force);
	setRenamings(newNames);
}
/**
 * @see MultiOperation
 */
protected String getMainTaskName() {
	return Util.bind("operation.renameElementProgress"); //$NON-NLS-1$
}
/**
 * @see CopyElementsOperation#isRename()
 */
protected boolean isRename() {
	return true;
}
/**
 * @see MultiOperation
 */
protected IJavaModelStatus verify() {
	IJavaModelStatus status = super.verify();
	if (! status.isOK())
		return status;
	if (this.renamingsList == null || this.renamingsList.length == 0)
		return new JavaModelStatus(IJavaModelStatusConstants.NULL_NAME);
	return JavaModelStatus.VERIFIED_OK;
}
/**
 * @see MultiOperation
 */
protected void verify(IJavaElement element) throws JavaModelException {
	int elementType = element.getElementType();
	
	if (element == null || !element.exists())
		error(IJavaModelStatusConstants.ELEMENT_DOES_NOT_EXIST, element);
		
	if (element.isReadOnly())
		error(IJavaModelStatusConstants.READ_ONLY, element);
		
	if (!(element instanceof ISourceReference))
		error(IJavaModelStatusConstants.INVALID_ELEMENT_TYPES, element);
		
	if (elementType < IJavaElement.TYPE || elementType == IJavaElement.INITIALIZER)
		error(IJavaModelStatusConstants.INVALID_ELEMENT_TYPES, element);
		
	Member localContext;
	if (element instanceof Member && (localContext = ((Member)element).getOuterMostLocalContext()) != null && localContext != element) {
		// JDOM doesn't support source manipulation in local/anonymous types
		error(IJavaModelStatusConstants.INVALID_ELEMENT_TYPES, element);
	}

	verifyRenaming(element);
}
}
