package core.btree;

import java.io.OutputStream;

import util.IntPushOperator;

/**
 * Represents a node from the b-tree (either directory node or leaf).
 * 
 * @author jens/marcos
 */
public interface BTreeNode {

	/**
	 * Indicates if this node is a leaf.
	 * 
	 * @return
	 */
	public boolean isLeaf();

	/**
	 * Obtains all values mapped to the given key. Values are delivered to the
	 * provided push operator.
	 * 
	 * @param key
	 * @param results
	 */
	public void get(int key, IntPushOperator results);

	/**
	 * Obtains all values mapped to the given key range (low and high,
	 * inclusive). Values are delivered to the provided push operator.
	 * 
	 * @param lowKey
	 * @param highKey
	 * @param results
	 */
	public void queryRange(int lowKey, int highKey, IntPushOperator results);

	/**
	 * Adds the given mapping to the node.
	 * 
	 * @param key
	 * @param value
	 * @return
	 */
	public SplitInfo add(int key, int value, int lowKey, int highKey, LeafCarrier leafCarrier);

	/**
	 * Removes a single instance of the key-value mapping from the node. If the
	 * value given is equal to BTreeConstants.ALL_MAPPINGS then all mappings
	 * associated to the given key will be removed.
	 * 
	 * @param key
	 * @param lowKey
	 * @param highKey
	 */
	public void remove(int key, int value, int lowKey, int highKey);

	/**
	 * Indicates if this node is empty.
	 * 
	 * @return
	 */
	public boolean isEmpty();

	/**
	 * Serializes this node to a dotty representation (to be rendered with
	 * graphviz).
	 * 
	 * @param dest
	 */
	public void toDot(OutputStream dest);

	// do we need this method?
	// public boolean update(int key, int value);
}