package core.csbtree;


/**
 * Helper for csb-tree nodes.
 * 
 * @author Yagiz
 *
 */
public class NodeGroupArrayMap {

	/** In the beginning m keys stored on this node group and then it will be expanded */
	protected int[] keys;

	/**
	 * In the beginning 1 pointer stored in this node group and then it will be expanded
	 */
	protected CSBTreeNode[] nodes;

	/**
	 * Stores the number of current keys for each node.
	 */
	protected int[] currentSizes;
	
	/**
	 * Number of the keys between two pointers of a node (i.e 4*k)
	 */
	protected int m;
	
	/**
	 * Actual Number of nodes in the node group
	 */
	protected int numberOfNodes = 0;
	
	/**
	 * Creates an empty node group with no key and only one node in it.
	 * 
	 * @param m - number of the keys in a node (i.e. capacity)
	 */
	public NodeGroupArrayMap(int m) 
	{
		this.m = m;
		keys = new int[m];
		numberOfNodes = 1;
		nodes = new CSBTreeNode[1];
		currentSizes = new int[1];
		
		nodes[0] = NodeGroup.NULL;
		currentSizes[0] = 0;
	}
	
	public static int binarySearch(int[] a, int key, int from, int to) {
		int low = from;
		int high = to;

		for (; low <= high;) {
			int mid = (low + high) >> 1;
			long midVal = a[mid];

			if (midVal < key)
				low = mid + 1;
			else if (midVal > key)
				high = mid - 1;
			else
				return mid; // key found

		}
		return -(low + 1); // key not found.
	}
	
	/**
	 * Creates an empty node group with no keys, no pointers and only
	 * specified number of nodes in it.
	 * 
	 * @param m - number of the keys in a node (i.e. capacity)
	 * @param numberOfNodes - actual number of nodes in the node group
	 */
	public NodeGroupArrayMap(int m, int numberOfNodes)
	{
		this.m = m;
		this.numberOfNodes = numberOfNodes;
		keys = new int[m*numberOfNodes];
		nodes = new CSBTreeNode[numberOfNodes];
		currentSizes = new int[numberOfNodes];
	}

	
	
	/**
	 * Obtains the position in the key that represents the interval in
	 * which the provided key falls. The node that the key can be found is
	 * shown by index.
	 * 
	 * @param index - specifies the node in the node group
	 * @param key - the key of the tuple whose position has to be found in
	 * the node group
	 * @return - the position (interval) of the key in the node group
	 */
	public int getIntervalPosition(int key, int index) 
	{
		if (currentSizes[index] == 0) {
			return -1;
		} else {
			// index says in which node we can find the interval.
			// since we have m keys in a node, index*m makes us
			// start directly from the node requested
			int pos = binarySearch(keys, key, index*m,
					index*m + currentSizes[index] - 1);

			// we are left-aligned, so we take equal to the right, non-equal at
			// insertion point
			if (pos < 0) {
				// key not found: calculate insertion point
				pos = -(pos + 1);
			} else {
				// key found: take right path
				pos++;
			}
			
			return pos;
		}
	}
	
	/**
	 * Puts the given key just in the exact position specified, shift the
	 * others accordingly.
	 * 
	 * @param key - the key of the tuple which has to be put in
	 * the node group
	 * @param pos - specifies the position in the node.
	 * @param index - specifies the node in the node group
	 */
	public void put(int key, int pos, int index) {
		
		int indexXm = index * m;
		
		if (pos < currentSizes[index]) 
		{
			System.arraycopy(keys, indexXm + pos, keys, indexXm + pos + 1, currentSizes[index] - pos);
			keys[indexXm + pos] = key;
			currentSizes[index]++;
		}
		else 
		{
			keys[indexXm + currentSizes[index]] = key;
			currentSizes[index]++;
		}
		
		
		
	}


	/**
	 * Splits a node into two nodes in a node group. Returns the pivot that
	 * has to be passed to the parent. 
	 * 
	 * @param pivot - the key that has to be added into that node and that
	 * comes from the child
	 * @param pos - absolute position of the key in the nodegroup not in the node
	 * @param index - specifies the node in the node group
	 * @param rightchild - the node group that the pointer of the new node
	 * has to point
	 * @return - the pivot that has to be sent to parent. Before the Node splits
	 * it is the key in the middle of the node.
	 */
	public int splitNode(int pivot, int pos, int index, core.csbtree.CSBTreeNode rightchild) 
	{
		++numberOfNodes;
		
		int posInNode = pos % m;
		
		if(pos == (index+1)*m )
		{
			posInNode = m;
		}
		
		int[] newKeys = new int[numberOfNodes * m];
		core.csbtree.CSBTreeNode[] newNodes = new CSBTreeNode[numberOfNodes];
		int[] newCurrentSizes = new int[numberOfNodes];
		
		int addPivotTo = 0; //0: new left node, 1: splitInfo returned, 2: new right node
		int splitPosInNode = (currentSizes[index] + 1) >> 1;
		if(posInNode > splitPosInNode)
		{
			splitPosInNode++;
			addPivotTo = 2;
		}
		else if (posInNode == splitPosInNode)
		{
			addPivotTo = 1;
		}
		
		int splitPos = index * m + splitPosInNode;
		
		//copy the keys
		if(addPivotTo == 1)
			System.arraycopy(keys, 0, newKeys, 0, splitPos);
		else
			System.arraycopy(keys, 0, newKeys, 0, splitPos - 1);
		System.arraycopy(keys, splitPos, newKeys, (index + 1) * m, currentSizes[index] - splitPosInNode);
		System.arraycopy(keys, (index + 1) *m, newKeys, (index + 2) * m, (numberOfNodes - index - 2) * m);
		
		//copy the pointers
		System.arraycopy(nodes, 0, newNodes, 0, index + 1);
		System.arraycopy(nodes, index + 1, newNodes, index + 2, numberOfNodes - index - 2);
		newNodes[index + 1] = rightchild;
		
		//copy currentSizes
		System.arraycopy(currentSizes, 0, newCurrentSizes, 0, index + 1);
		System.arraycopy(currentSizes, index + 1, newCurrentSizes, index + 2, numberOfNodes - index - 2);
		if(addPivotTo == 1)
			newCurrentSizes[index] = splitPosInNode;
		else
			newCurrentSizes[index] = splitPosInNode - 1;
		newCurrentSizes[index + 1] = currentSizes[index] - splitPosInNode;
		
		int newPivot = keys[splitPos-1];
		
		//assign new values to the olds
		keys = newKeys;
		nodes = newNodes;
		currentSizes = newCurrentSizes;
		
		switch (addPivotTo) {
		case 0:
			put(pivot, posInNode, index);
			return newPivot;
		case 2:
			posInNode = posInNode - splitPosInNode;
			put(pivot, posInNode, index + 1);
			return newPivot;
		default:
			return pivot;
				
		}
		
	}

	/**
	 * Indicates if the node is full.
	 * 
	 * @param index - specifies the node in the node group
	 * @return
	 */
	public boolean isNodeFull(int index)
	{
		if(currentSizes[index] < m)
			return false;
		return true;
	}

	/**
	 * Splits the node group into two node groups
	 * 
	 * @return - the right child (i.e the newly created node group) that has to be
	 * passed to the parent.
	 */
	public NodeGroupArrayMap splitNodeGroup() 
	{
		int numberOfNodesInLeftChild = (numberOfNodes + 1) / 2;
		int numberOfNodesInRightChild = numberOfNodes - numberOfNodesInLeftChild;
		
		int[] newKeys = new int[numberOfNodesInLeftChild * m];
		CSBTreeNode[] newNodes = new CSBTreeNode[numberOfNodesInLeftChild];
		int[] newCurrentSizes = new int[numberOfNodesInLeftChild];
		
		NodeGroupArrayMap rightchild = new NodeGroupArrayMap(m, numberOfNodesInRightChild);
		
		//copy the keys
		System.arraycopy(keys, 0, newKeys, 0, numberOfNodesInLeftChild*m);
		System.arraycopy(keys, numberOfNodesInLeftChild*m, rightchild.keys, 0, numberOfNodesInRightChild*m);
		
		//copy the pointers
		System.arraycopy(nodes, 0, newNodes, 0, numberOfNodesInLeftChild);
		System.arraycopy(nodes, numberOfNodesInLeftChild, rightchild.nodes, 0, numberOfNodesInRightChild);
		
		//copy currentSizes
		System.arraycopy(currentSizes, 0, newCurrentSizes, 0, numberOfNodesInLeftChild);
		System.arraycopy(currentSizes, numberOfNodesInLeftChild, rightchild.currentSizes, 0, numberOfNodesInRightChild);
		
		//assign new values to the olds
		keys = newKeys;
		nodes = newNodes;
		currentSizes = newCurrentSizes;
		numberOfNodes = numberOfNodesInLeftChild;
		
		return rightchild;
		
	}

	
	public String toString() {
		StringBuffer sb = new StringBuffer();

		for(int i = 0; i < numberOfNodes; i++)
		{
			String nodeValue = nodes[i] == NodeGroup.NULL ? "NULL | " : Integer
					.toString(nodes[i].hashCode()) + " | ";
			sb.append(nodeValue);
			for (int j = 0; j < currentSizes[i]; j++) 
			{
				if(j == currentSizes[i] - 1)
				{
					sb.append(keys[j + i * m] + "/ ");
					break;
				}
				sb.append(keys[j + i * m] + ".");
	
			}
		}

		return sb.toString();
	}
	
	
	
}
