package core.csbtree;

import java.io.OutputStream;

import util.IntPushOperator;

/**
 * Represents a csb-tree directory node.
 * 
 * @author Yagiz
 *
 */
public class NodeGroup implements CSBTreeNode{

	protected static final NodeGroup NULL = new NodeGroup(0, null);
	
	protected NodeGroupArrayMap entries;

	protected int k;
	
	
	/**
	 * Creates a new NodeGroup object from an existing NodeGroupArrayMap
	 * 
	 * @param k - 1/4 of the number of keys between two pointers of a
	 * node group i.e. (m = 4k)
	 * @param entries - an existing NodeGroupArrayMap
	 */
	public NodeGroup(int k, NodeGroupArrayMap entries) 
	{
		this.k = k;
		this.entries = entries;
	}
	
	
	/**
	 * Creates a NodeGroup which has only one key and one pointer in the beginning
	 * 
	 * @param child - the only child of the node group which will be created
	 * @param pivot - the only key of the node group which will be created
	 * @param k - 1/4 of the number of keys between two pointers of a node
	 */
	public NodeGroup(CSBTreeNode child, int pivot, int k) 
	{
		this.k = k;
		
//		at first time the node group has only one node and one pointer for that node.
		this.entries = new NodeGroupArrayMap(4*k);

		entries.nodes[0] = child;
		entries.keys[0] = pivot;
		entries.currentSizes[0] = 1;
		
	}
	
	
	/**
	 * Adds the given mapping to the NodeGroup. Recursively called by the children.
	 * The node in which the key will be inserted in the nodegroup is determined
	 * by the index. If a non-null splitInfo comes from the child, the node splits.
	 * In addition to that condition, if the parent is full, then the node group
	 * itself also splits.
	 * 
	 * 
	 * @param index - specifies the node in the node group
	 * @param key - the key of the tuple which has to be inserted into the tree
	 * @param value - the value of the tuple which has to be inserted into the tree
	 * @param isParentNodeFull - specifies whether the parent node (i.e. the node
	 * inside the caller node group) is full or not
	 * @param lowKey - lower bound for the key
	 * @param highKey - upper bound for the key
	 * @param leafCarrier - useless
	 * @return - a null splitInfo if no split is needed/done 
	 * or a splitInfo with only null rightChild if only a node split is needed/done
	 * or a full splitInfo if a node group split is also needed/done
	 */
	public SplitInfo add(int index, int key, int value, boolean isParentNodeFull, int lowKey, int highKey, LeafCarrier leafCarrier) 
	{			
		
		//get next node to call insertion.
		//if we are in the root we always only have one node.
		int pos = entries.getIntervalPosition(key, index);
		core.csbtree.CSBTreeNode next = entries.nodes[index];

		// calculate key interval for next node
		int newLowKey = pos == 0 ? lowKey : entries.keys[pos-1];
		int newHighKey = pos == ((entries.numberOfNodes-1)*4*k + entries.currentSizes[entries.numberOfNodes-1]) ? highKey : entries.keys[pos];
		
		//pos value between 0 and m needed to use as an index in the child
		int newIndex = pos % (4*k);
		
		if(pos == (index+1)*4*k )
		{
			newIndex = 4*k;
		}
		
		SplitInfo splitInfo = next.add(newIndex, key, value, entries.isNodeFull(index), newLowKey, newHighKey, leafCarrier);
		
		//after recursion, check for split coming from lower levels
		if (splitInfo != null) {
			if(splitInfo.rightNode == null)
			{
				// this means our node is not full and we can add it. If it were full, a fully specified splitInfo would come.
				entries.put(splitInfo.pivot, pos % (4*k), index);
			}
			else
			{
				return split(splitInfo.pivot, splitInfo.rightNode, isParentNodeFull, index);
			}
		}
		
		return null;
	}

	
	/**
	 * Splits the full node into two nodes first. Then if the parent is full, also
	 * splits the node group.
	 * 
	 * @param isParentNodeFull - specifies whether the parent node (i.e. the node
	 * insidethe caller node group) is full or not
	 * @param rightchild - the new node/leaf group appears after the split
	 * @param pivot - the first key of the new node/leaf that appears after the split
	 * of the node/leaf
	 * @param index - specifies the node in the node group
	 * @return - a splitInfo with only null rightChild if only a node split is
	 * needed/done (i.e. isParentNodeFull is false) or a full splitInfo if also a node
	 * group split is needed/done (i.e. isParentNodeFull is true)
	 */
	private core.csbtree.SplitInfo split(int pivot, core.csbtree.CSBTreeNode rightchild, boolean isParentNodeFull, int index) 
	{
		// we have to split the node for all conditions
		int pos = entries.getIntervalPosition(pivot, index);
		int nextPivot = entries.splitNode(pivot, pos, index, rightchild);
		
		if(!isParentNodeFull)
		{
			// simply move the pivot as a key for the parent
			return new core.csbtree.SplitInfo(this, nextPivot, null);
		}
		
		// also split the node group:
		NodeGroup newNode = new NodeGroup(k, entries.splitNodeGroup());

		return new core.csbtree.SplitInfo(this, nextPivot, newNode);
	}
	
	
	/**
	 * Removes a single instance of the key-value mapping from the node
	 * calculated by the index. If the value given is equal to
	 * BTreeConstants.ALL_MAPPINGS then all mappings associated to the given key will
	 * be removed. Recursively called by the children. The node in which the key will
	 * be inserted in the nodegroup is determined by the index.
	 * 
	 * @param index - specifies the node in the node group
	 * @param key - the key of the tuple which has to be removed from the tree
	 * @param value - the value of the tuple which has to be removed from the tree
	 * @param lowKey - lower bound for the key
	 * @param highKey - upper bound for the key
	 */
	public void remove(int index, int key, int value, int lowKey, int highKey) 
	{
		//get next node to recurse deletion
		int pos = entries.getIntervalPosition(key, index);
		core.csbtree.CSBTreeNode next = entries.nodes[index];
		
		//pos value between 0 and m needed to use as a index in the child
		int newIndex = pos % (4*k);
		
		if(pos == (index+1)*4*k )
		{
			newIndex = 4*k;
		}

		if (next != null || pos != -1) 
		{
			// calculate key interval for next node
			int newLowKey = pos == 0 ? lowKey : entries.keys[pos-1];
			int newHighKey = pos == ((entries.numberOfNodes-1)*4*k + entries.currentSizes[entries.numberOfNodes-1]) ?
					highKey : entries.keys[pos];
			next.remove(newIndex, key, value, newLowKey, newHighKey);
		}
	}
	
	
	/**
	 * Obtains all values mapped to the given key in the node specified
	 * by the index. Recursively called by the children. Values are delivered to the
	 * provided push operator.
	 * 
	 * @param index - specifies the node in the node group
	 * @param key - the key of the tuple which has to be searched and found in the tree
	 * @param results - the consumer which is used after finding the mapping.
	 */
	public void get(int index, int key, IntPushOperator results) 
	{
		//get next node to recurse
		int pos = entries.getIntervalPosition(key, index);
		core.csbtree.CSBTreeNode next = entries.nodes[index];
		
		//pos value between 0 and m needed to use as a index in the child
		int newIndex = pos % (4*k);
		
		if(pos == (index+1)*4*k )
		{
			newIndex = 4*k;
		}
		
		if (next != null || pos != -1)
			next.get(newIndex, key, results);
	}
	
	
	/**
	 * Obtains all values mapped to the given key range (low and high,
	 * inclusive) and index. Recursively called by the children. Values are delivered
	 * to the provided push operator.
	 * 
	 * @param index - specifies the node in the node group
	 * @param lowKey - start key of the range
	 * @param highKey - last key of the range
	 * @param results - the consumer which is used after finding the range.
	 */
	public void queryRange(int index, int lowKey, int highKey, IntPushOperator results) 
	{
		//get next node to recurse
		int pos = entries.getIntervalPosition(lowKey, index);
		core.csbtree.CSBTreeNode next = entries.nodes[index];
		
		//pos value between 0 and m needed to use as a index in the child
		int newIndex = pos % (4*k);
		
		if(pos == (index+1)*4*k )
		{
			newIndex = 4*k;
		}
		
		if (next != null || pos != -1)
			next.queryRange(newIndex, lowKey, highKey, results); 
	}
	
	
	/**
	 * Indicates if this node is a leaf.
	 * 
	 * @return
	 */
	public boolean isLeaf() 
	{
		return false;
	}
	
	/**
	 * Indicates if this node group is empty.
	 * 
	 * @return
	 */
	public boolean isEmpty() 
	{
		if(entries.numberOfNodes == 0)
			return true;
		return false;
	}

	/**
	 * Serializes this node to a dotty representation (to be rendered with
	 * graphviz).
	 * 
	 * @param dest
	 */
	public void toDot(OutputStream dest) 
	{
		
	}
	
	
	public boolean isFull(int index)
	{
		return entries.currentSizes[index] == 4*k;
	}
	
	public String toString() {
		return "[" + entries.toString() + "]";
	}
	
}
