package core.csbtree;

import util.IntPushOperator;

/**
 * Helper for csb-tree leaf groups.
 * 
 * @author alekh
 */
public class LeafGroupArrayMap {
    // leaf parameter
    protected int two_k_star;    // single keys array for all leaves in this leaf group
    protected int[] keys;    // single values array for all leaves in this leaf group
    protected int[] values;    // total capacity of all leaves currently in this leaf groups
    protected int currentGroupSize = 0;    // array for storing the current sizes of each of the leaves in this leaf group.
    protected int[] currentLeafSize;    // last filled position in the keys/values array
    protected int currentSize = 0;
    protected static final int STOP = 0;
    protected static final int CONTINUE_WITH_BINSEARCH = 1;
    protected static final int CONTINUE_WITH_SCAN = 2;
    // binary search method
    public static int binarySearch(int[] a, int key, int from, int to) {
        int low = from;
        int high = to;

        for (; low <= high;) {
            int mid = (low + high) >> 1;
            long midVal = a[mid];

            if (midVal < key) {
                low = mid + 1;
            } else if (midVal > key) {
                high = mid - 1;
            } else {
                // key found: search for first occurrence linearly
                // this search is necessary in the presence of duplicates
                int pos = mid - 1;
                while (pos >= from && a[pos] == key) {
                    pos--;
                }
                // return last valid position
                return pos + 1;
            }

        }
        return -(low + 1); // key not found.
    }

    /**
     * Create new leaf group entries. 
     * @param n - size of leaf i.e. (2k_star)
     */
    public LeafGroupArrayMap(int n) {
        two_k_star = n;
        keys = new int[n];
        values = new int[n];
        currentLeafSize = new int[1];
        currentGroupSize = two_k_star;
    }

    /**
     * Add a key to the leaf group.
     * @param leafNumber - the number of the leaf to insert.
     * @param key - key to insert.
     * @param value - value of the corresponding key.
     * @return - true if add needs to be tried in next leaf group
     * 			 false if insertion point was found and insertion done.
     */
    public boolean tryAdd(int leafNumber, int key, int value) {
        // check for entire leaf group scan
        int leafStartPosition = (leafNumber - 1) * two_k_star;
        int leafEndPosition = leafStartPosition + currentLeafSize[leafNumber - 1] - 1;

        if (currentLeafSize[leafNumber - 1] == 0) {
            // insert in this node: we assume overflow leaves are
            // garbage-collected on deletions
            keys[leafStartPosition] = key;
            values[leafStartPosition] = value;
            currentLeafSize[leafNumber - 1]++;
            if (leafNumber == currentLeafSize.length) {
                currentSize++;
            }
            return false;
        } else {
            int pos = binarySearch(keys, key, leafStartPosition, leafEndPosition);
            if (pos < 0) {
                // calculate insertion point
                pos = -(pos + 1);
            }

            // find insertion point
            while (pos <= leafEndPosition && keys[pos] <= key) {
                pos++;
            }
            if (pos == currentGroupSize) {
                // continue search for insertion point in next leaf group
                return true;
            } else if (pos == leafNumber * two_k_star) {
                // continue search for insertion point in next leaf
                return tryAdd(leafNumber + 1, key, value);
            } else {
                // well, the insertion point is here
                addAtPos(key, value, leafNumber, pos);
                return false;
            }
        }
    }

    /**
     * Add to the first leaf of the leaf group.
     * @param key - key to insert
     * @param value - value of the corresponding key.
     * @return - true if add needs to be tried in next leaf group
     * 			 false if insertion point was found and insertion done.
     */
    public boolean tryAdd(int key, int value) {
        return tryAdd(1, key, value);
    }

    /**
     * Get the array index for the given position of the given leaf number. 
     * @param leafNumber
     * @param position - inside the leaf
     * @return - index position
     */
    public int indexOf(int leafNumber, int position) {
        return two_k_star * (leafNumber - 1) + position - 1;
    }

    /**
     * Get the number of leaves currently in the leaf group.
     * @return
     */
    public int leaves() {
        return currentLeafSize.length;
    }

    /**
     * Get the total capacity of all leaves currently in this leaf groups.
     * @return
     */
    public int size() {
        return currentGroupSize;
    }

    /**
     * Get the number of filled elements in a leaf.
     * @param leafNumber - number of the leaf
     * @return - number of filled elements.
     */
    public int size(int leafNumber) {
        return currentLeafSize[leafNumber - 1];
    }

    /**
     * Get the total number of keys in the leaf group.
     * @return
     */
    public int getKeyNumber() {
        int count = 0;
        for (int i = 0; i < currentLeafSize.length; i++) {
            count += currentLeafSize[i];
        }
        return count;
    }

    /**
     * Add key at a position 
     * @param key - key to be added
     * @param value - value of the corresponding key
     * @param leafNumber - number (starting from 1) of the leaf where the addition has to be done.
     * @param pos - position to add at
     */
    public void addAtPos(int key, int value, int leafNumber, int pos) {
        int leafLastIndex = indexOf(leafNumber, currentLeafSize[leafNumber - 1]);
        if (pos <= leafLastIndex) {
            System.arraycopy(keys, pos, keys, pos + 1, leafLastIndex - pos + 1);
            System.arraycopy(values, pos, values, pos + 1, leafLastIndex - pos + 1);
            currentLeafSize[leafNumber - 1]++;
            if (leafNumber == currentLeafSize.length) {
                currentSize++;
            }
            keys[pos] = key;
            values[pos] = value;
        } else {
            keys[leafLastIndex + 1] = key;
            values[leafLastIndex + 1] = value;
            currentLeafSize[leafNumber - 1]++;
            if (leafNumber == currentLeafSize.length) {
                currentSize++;
            }
        }
    }

    /**
     * Split a leaf into two leaves. 
     * Reditribute elements to both leaves.
     * Expand the leaf group to accomodate the new leaf.
     * @param leafNumber - number (starting from 1) of the leaf to be splitted.
     */
    public void split(int leafNumber) {
        // split the leaf
        int leafEnd = indexOf(leafNumber, currentLeafSize[leafNumber - 1]);
        final int mid = indexOf(leafNumber, (currentLeafSize[leafNumber - 1] / 2) + 1);

        int[] newkeys = new int[currentGroupSize + two_k_star];
        int[] newvalues = new int[currentGroupSize + two_k_star];
        int[] newCurrentLeafSize = new int[leaves() + 1];

        // expand keys and values array to accomodate the new leaf
        System.arraycopy(keys, 0, newkeys, 0, mid);
        System.arraycopy(keys, mid, newkeys, leafEnd + 1, two_k_star - (mid % two_k_star));
        System.arraycopy(keys, leafEnd + 1, newkeys, leafEnd + 1 + two_k_star, currentGroupSize - leafEnd - 1);

        System.arraycopy(values, 0, newvalues, 0, mid);
        System.arraycopy(values, mid, newvalues, leafEnd + 1, two_k_star - (mid % two_k_star));
        System.arraycopy(values, leafEnd + 1, newvalues, leafEnd + 1 + two_k_star, currentGroupSize - leafEnd - 1);

        // expand currentLeafSize array and adjust the sizes of the two leaves.
        System.arraycopy(currentLeafSize, 0, newCurrentLeafSize, 0, leafNumber - 1);
        System.arraycopy(currentLeafSize, leafNumber, newCurrentLeafSize, leafNumber + 1, leaves() - leafNumber);

        newCurrentLeafSize[leafNumber - 1] = currentLeafSize[leafNumber - 1] / 2;
        newCurrentLeafSize[leafNumber] = currentLeafSize[leafNumber - 1] - currentLeafSize[leafNumber - 1] / 2;

        this.keys = newkeys;
        this.values = newvalues;
        this.currentLeafSize = newCurrentLeafSize;
        currentSize = currentGroupSize + currentLeafSize[currentLeafSize.length - 1];
        currentGroupSize = currentGroupSize + two_k_star;

    }

    /**
     * Split the leaf group.
     * Redistribute leaves to both leaf groups.
     * Resize the leaf groups to just accomodate their respective leaves. 
     * @return - pointer to newly created leaf group.
     */
    public LeafGroupArrayMap split() {

        int midLeaf = (leaves() + 3) / 2;
        final int mid = indexOf(midLeaf, 1);
        LeafGroupArrayMap newMap = new LeafGroupArrayMap(two_k_star);

        newMap.currentLeafSize = new int[leaves() - midLeaf + 1];
        newMap.keys = new int[currentGroupSize - mid];
        newMap.values = new int[currentGroupSize - mid];
        System.arraycopy(keys, mid, newMap.keys, 0, currentSize - mid);
        System.arraycopy(values, mid, newMap.values, 0, currentSize - mid);
        System.arraycopy(currentLeafSize, midLeaf - 1, newMap.currentLeafSize, 0, leaves() - midLeaf + 1);
        newMap.currentGroupSize = (leaves() - midLeaf + 1) * two_k_star;
        newMap.currentSize = newMap.currentGroupSize - two_k_star + newMap.currentLeafSize[newMap.currentLeafSize.length - 1];

        // NOTE: this operation may be CPU expensive!
        int[] newKeys = new int[(midLeaf - 1) * two_k_star];
        int[] newValues = new int[(midLeaf - 1) * two_k_star];
        int[] newCurrentLeafSize = new int[midLeaf - 1];
        System.arraycopy(keys, 0, newKeys, 0, mid);
        System.arraycopy(values, 0, newValues, 0, mid);
        System.arraycopy(currentLeafSize, 0, newCurrentLeafSize, 0, midLeaf - 1);
        keys = newKeys;
        values = newValues;
        currentLeafSize = newCurrentLeafSize;
        currentGroupSize = (midLeaf - 1) * two_k_star;
        currentSize = currentGroupSize - two_k_star + currentLeafSize[currentLeafSize.length - 1];

        return newMap;
    }

    /**
     * Remove key from a leaf 
     * @param leafNumber - number (starting from 1) of the leaf to remove from.
     * @param key - key to be removed
     * @param value - value of the corresponding key.
     * @return - true if next leaf group needs to be tried.
     * 			 false if removal point was found.
     */
    public boolean remove(int leafNumber, int key, int value) {
        if (currentGroupSize == 0) {
            // continue search
            return true;
        }
        int leafStartPosition = (leafNumber - 1) * two_k_star;
        int leafEndPosition = leafStartPosition + currentLeafSize[leafNumber - 1] - 1;
        int pos = binarySearch(keys, key, leafStartPosition, leafEndPosition);
        if (pos < 0) {
            // key does not exist here, check if we should go to next array
            pos = -(pos + 1);
            if (pos == currentSize) {
                // try next leaf group
                return true;
            } else if (pos == leafEndPosition + 2) {
                // try next leaf
                return remove(leafNumber + 1, key, value);
            } else {
                return true;
            }
        } else {
            // key exists, delete:
            // first find occurrence range of key in this array
            int firstOccurrence = -1; // pos
            int lastOccurrence;

            while (pos <= leafEndPosition && keys[pos] == key) {

                if (value == CSBTreeConstants.ALL_MAPPINGS) {
                    // mark first occurrence
                    if (firstOccurrence == -1) {
                        firstOccurrence = pos;
                    }
                } else {
                    if (values[pos] == value) {
                        // found desired mapping: only that mapping should be
                        // removed
                        firstOccurrence = pos;
                        break;
                    }
                }
                // continue scanning
                pos++;
            }
            // fix last occurrence
            if (value == CSBTreeConstants.ALL_MAPPINGS) {
                lastOccurrence = pos - 1;
            } else {
                lastOccurrence = firstOccurrence;
            }
            boolean continueSearch = (value == CSBTreeConstants.ALL_MAPPINGS) || (firstOccurrence == -1) ? (pos == leafEndPosition + 1) : false;

            // now delete all occurrences in one move, if necessary
            if (firstOccurrence != -1) {
                System.arraycopy(keys, lastOccurrence + 1, keys,
                        firstOccurrence, leafEndPosition - lastOccurrence);
                System.arraycopy(values, lastOccurrence + 1, values,
                        firstOccurrence, leafEndPosition - lastOccurrence);
                currentLeafSize[leafNumber - 1] -= (lastOccurrence - firstOccurrence + 1);
                currentSize -= (lastOccurrence - firstOccurrence + 1);
            }

            return continueSearch;
        }
    }

    /**
     * Remove key from first leaf of leaf group.
     * @param key - key to be removed
     * @param value - value of the corresponding key.
     * @return - true if next leaf group needs to be tried.
     * 			 false if removal point was found.
     */
    public boolean remove(int key, int value) {
        return remove(1, key, value);
    }

    /**
     * Query key from a leaf.  
     * @param leafNumber - number (starting from 1) of the leaf to query.
     * @param key - key query
     * @param results - result buffer
     * @return
     */
    public int get(int leafNumber, int key, IntPushOperator results) {
        int leafStartPosition = (leafNumber - 1) * two_k_star;
        int leafEndPosition = leafStartPosition + currentLeafSize[leafNumber - 1] - 1;

        if (currentSize == 0) {
            // continue search to next leaf group
            return CONTINUE_WITH_BINSEARCH;
        } else if (currentLeafSize[leafNumber - 1] == 0) {
            // continue search to next leaf
            return get(leafNumber + 1, key, results);
        } else {
            int pos = binarySearch(keys, key, leafStartPosition, leafEndPosition);
            if (pos < 0) {
                // key not found: if we are at the end of the array, maybe key
                // is in the next one
                pos = -(pos + 1);
                if (pos == currentSize) {
                    return CONTINUE_WITH_BINSEARCH;
                } else if (pos == leafEndPosition + 2) {
                    return get(leafNumber + 1, key, results);
                } else {
                    return STOP;
                }
            } else {
                // get values corresponding to key				
                return continueGet(pos, key, results);
            }
        }
    }

    /**
     * Query key in the first leaf of the leaf group.
     * @param key - key query
     * @param results - result buffer
     * @return
     */
    public int get(int key, IntPushOperator results) {
        return get(1, key, results);
    }

    /**
     * Key found. Keep buffering to handle duplicates.
     * @param pos - first position
     * @param key - key being scanned.
     * @param results - result buffer
     * @return
     */
    public int continueGet(int pos, int key, IntPushOperator results) {
        while (pos < currentSize && keys[pos] == key) {
            results.pass(values[pos]);
            pos++;
        }
        return pos == currentSize ? CONTINUE_WITH_SCAN : STOP;
    }

    /**
     * Query a range of keys starting from a leaf of this leaf group.
     * @param leafNumber - number (starting from 1) of the leaf to start.
     * @param lowKey - lower bound of the query range
     * @param highKey - upper bound of the query range
     * @param results - result buffer
     * @return 0 - if upper bound is exceeded
     * 		   1 - if scan had started and end of leaf group is encountered
     * 		   2 - if scan hadn't started and end of leaf group is encountered
     */
    public int queryRange(int leafNumber, int lowKey, int highKey, IntPushOperator results) {
        int leafStartPosition = (leafNumber - 1) * two_k_star;
        int leafEndPosition = leafStartPosition + currentLeafSize[leafNumber - 1] - 1;

        if (currentSize == 0) {
            return CONTINUE_WITH_BINSEARCH; // maybe leaf was emptied by
        // deletions
        } else if (currentLeafSize[leafNumber - 1] == 0) {
            // try next leaf in this leaf group
            return queryRange(leafNumber + 1, lowKey, highKey, results);
        } else {
            int pos = binarySearch(keys, lowKey, leafStartPosition, leafEndPosition);
            if (pos < 0) {
                // key not found: get starting point for scan
                pos = -(pos + 1);
            }

            // scan from given position onwards
            return continueScan(leafNumber, pos, highKey, results);
        }
    }

    /**
     * Query a range of keys starting from the first leaf of this leaf group.
     * @param lowKey - lower bound of the query range
     * @param highKey - upper bound of the query range
     * @param results - result buffer
     * @return
     */
    public int queryRange(int lowKey, int highKey, IntPushOperator results) {
        return queryRange(1, lowKey, highKey, results);
    }

    /**
     * Lower bound of the range query detected. Scan the leaves sequentially.
     * @param pos - position to start scan from
     * @param highKey - upper bound of the query range
     * @param results
     * @return - result buffer
     */
    public int continueScan(int leafNumber, int pos, int highKey, IntPushOperator results) {
        boolean returnedSomething = false;

        //if(currentSize-pos>0 && highKey>keys[currentSize-1]){
        //	int[] bulk_result = new int[currentSize-pos];
        //	System.arraycopy(keys, pos, bulk_result, 0, currentSize-pos);
        //	results.pass(bulk_result);
        //	return CONTINUE_WITH_SCAN;
        //}else{

        while (pos < currentLeafSize[leafNumber-1] && keys[pos] <= highKey) {
            results.pass(values[pos]);
            pos++;
            returnedSomething = true;
        }
        //}
        return pos == currentLeafSize[leafNumber-1] ? (returnedSomething ? CONTINUE_WITH_SCAN
                : CONTINUE_WITH_BINSEARCH) : STOP;
    }

    /**
     * String representation of this leaf group
     */
    @Override
    public String toString() {
        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < currentSize; i++) {
            sb.append(keys[i] + "=");
            sb.append(values[i]);
            if (i + 1 < currentSize) {
                sb.append(", ");
            }
        }
        return sb.toString();
    }
}
