package core.csbtree;

import java.io.OutputStream;

import util.IntPushOperator;

/**
 * A CSB tree leaf group implementation.
 * Each leaf group has leaves of size 2k* and there are
 * (4k+1) such logical leaves in a leaf group.
 * All leaves in a leaf group are stored in a single array.  
 * 
 * @author alekh
 */
public class LeafGroup implements CSBTreeNode{
	
	protected LeafGroupArrayMap entries;

	// node parameter
	protected int k;
	
	protected int k_star;

	// pointer to next leaf group
	protected LeafGroup nextLeafGroup;
	
	public LeafGroup(int k, int k_star, LeafGroupArrayMap entries) {
		
		this.k = k;
		this.k_star = k_star;
		this.entries = entries;
		this.nextLeafGroup = null;
	}
	
	/**
	 * Creates a leaf group.
	 * @param k - node parameter. Each node size is 4k
	 * @param k_star - leaf parameter. Each leaf size if 2k_star
	 */
	public LeafGroup(int k, int k_star) {
		this(k, k_star, new LeafGroupArrayMap(2*k_star));
	}
	
	/**
	 * Add a key to the leaf group.
	 * @param index - index of the leaf inside the leaf group i.e. (leaf number -1)
	 * @param key - key to be added
	 * @param value - value corresponding to the key
	 * @param isParentNodeFull - flag to indicate whether the corresponding parent node was full or not.
	 * 							 Used only in case the leaf splits.
	 * @param lowKey - the lower permissible value
	 * @param highKey - higher permissible value
	 * @param leafCarrier - to indicate which leaf the addition actually happened.
	 * @return null in case of no split.
	 * 		   left leaf group pointer and pivot in case of leaf split.
	 * 		   leaf and right leaf group pointers and pivot, in case of leaf group split. 
	 */
	public SplitInfo add(int index, int key, int value, boolean isParentNodeFull, int lowKey, int highKey, LeafCarrier leafCarrier){
		// search for insertion point: last allowed node of a scan for that key
		// a node is allowed as far as its first key is smaller than the highKey
		// this keeps the strict tree invariant that a leaf pointed to by
		// a given node only contains keys in the interval [lowKey, highKey).

		// HACK: even if a node is empty, we know that its array still contains
		// the key values it used to have when it was full. That allows us to
		// obtain the key range for a next leaf even if its entries are
		// completely deleted. This is necessary because we know that nodes
		// never get merged.
		boolean continueSearch = entries.tryAdd(index+1, key, value);
		LeafGroup currentLeafGroup = this;
		while (continueSearch && currentLeafGroup.nextLeafGroup != null
				&& currentLeafGroup.nextLeafGroup.entries.keys[0] < highKey) {
			continueSearch = currentLeafGroup.nextLeafGroup.entries.tryAdd(key, value);
			currentLeafGroup = currentLeafGroup.nextLeafGroup;
		}
		if (continueSearch) {
			// key was not added anywhere, so add it to this leaf anyway (no
			// more next leaves are available)
			int leafNumber = currentLeafGroup.entries.leaves();
			currentLeafGroup.entries.addAtPos(key, value, leafNumber, 2*k_star*(currentLeafGroup.entries.leaves()-1) + currentLeafGroup.entries.size(leafNumber));
		}

		// set leaf carrier, as appropriate
		if (leafCarrier != null) {
			leafCarrier.carriedLeaf = currentLeafGroup;
		}

		// invariant: key was added to currentLeaf
		if (currentLeafGroup.entries.size(index+1) >= (2*k_star)) {
			// split this leaf
			SplitInfo splitInfo = currentLeafGroup.split(index+1);
			if (isParentNodeFull){
				return currentLeafGroup.split(splitInfo);
			}
			return splitInfo;
		}		
		return null;
	}

	/**
	 * This is a split of CSB leaf. 
	 * A new leaf is created and the key are redistributed.
	 * @param leafNumber - the number of the leaf to be splitted. 
	 * @return - current leaf group as left pointer and pivot of the newly created leaf. 
	 */
	private SplitInfo split(int leafNumber) {
		//	split leaf into two new nodes:
		entries.split(leafNumber);
		
		// pivot is first entry in newLeaf keys
		int newLeafFirstIndex = entries.indexOf(leafNumber+1, 1);
		return new SplitInfo(this, entries.keys[newLeafFirstIndex], null);
	}
	
	/**
	 * This is a split of CSB leaf group.
	 * A new leaf group is created and the leaves are redistributed. 
	 * Split of indvidual leaves is to be handle while adding to the leaf group.
	 * NOTE: leaf group is splitted only if there was a split of an internal leaf.
	 * @param splitInfo - the splitting info after leaf split.
	 * @return - add newly created leaf group as right pointer to the split into and return.
	 */
	private SplitInfo split(SplitInfo splitInfo) {
		//	split leaf into two new nodes:
		LeafGroup newLeafGroup = new LeafGroup(k, k_star, entries.split());

		// make sure we point to this new sibling
		newLeafGroup.nextLeafGroup = nextLeafGroup;
		nextLeafGroup = newLeafGroup;

		return new SplitInfo(this, splitInfo.pivot, newLeafGroup);
	}

	/**
	 * Delete a key from the leaf group.
	 * @param index - index of the leaf inside the leaf group i.e. (leaf number -1)
	 * @param key - key to be removed
	 * @param value - value of the corresponding key
	 * @param lowKey - the lower permissible value
	 * @param highKey - higher permissible value
	 */
	public void remove(int index, int key, int value, int lowKey, int highKey) {
		// continue search on the leaf level until all entries with the given
		// key are guaranteed to be removed
		boolean continueSearch = entries.remove(index+1, key, value);
		LeafGroup currentLeafGroup = this;

		// HACK: even if a node is empty, we know that its array still contains
		// the key values it used to have when it was full. That allows us to
		// obtain the key range for a next leaf even if its entries are
		// completely deleted. This is necessary because we know that nodes
		// never get merged. (same as in insertion)
		// we use this information below to test if the nextLeaf is still in the
		// allowed range for this deletion
		while (continueSearch && currentLeafGroup.nextLeafGroup != null
				&& currentLeafGroup.nextLeafGroup.entries.keys[0] < highKey) {
			continueSearch = currentLeafGroup.nextLeafGroup.entries.remove(key, value);

			// check if we should garbage-collect currentLeaf.nextLeaf
			// that happens when it is an empty leaf not pointed to by a parent
			// this is true of all leaves visited in this loop
			if (currentLeafGroup.nextLeafGroup.entries.size() == 0) {
				currentLeafGroup.nextLeafGroup = currentLeafGroup.nextLeafGroup.nextLeafGroup;
			} else {
				// move on, there may be more stuff to remove
				currentLeafGroup = currentLeafGroup.nextLeafGroup;
			}
		}
	}
	
	/**
	 * Get a key from the leaf group.
	 * @param index - index of the leaf inside the leaf group i.e. (leaf number -1)
	 * @param key - key to be retrieved
	 * @param results - result buffer 
	 */
	public void get(int index, int key, IntPushOperator results) {
		// search in entries
		int continueSearch = entries.get(index+1, key, results);
		LeafGroup currentLeaf = nextLeafGroup;
		while (continueSearch != LeafGroupArrayMap.STOP && currentLeaf != null) {
			if (continueSearch == LeafGroupArrayMap.CONTINUE_WITH_BINSEARCH) {
				continueSearch = currentLeaf.entries.get(key, results);
			} else {
				continueSearch = currentLeaf.entries.continueGet(0, key,
						results);
			}
			currentLeaf = currentLeaf.nextLeafGroup;
		}
		results.thatsallfolks();
	}

	/**
	 * Execute a range query starting from this leaf group
	 * @param index - index of the leaf inside the leaf group i.e. (leaf number -1)
	 * @param lowKey - range start key
	 * @param highKey - range end key
	 * @param results - result buffer
	 */
	public void queryRange(int index, int lowKey, int highKey, IntPushOperator results) {
		// start with a query range on this leaf group and proceed to next leaf if
		// necessary
		int continueSearch = entries.queryRange(index+1, lowKey, highKey, results);
		LeafGroup currentLeaf = nextLeafGroup;
		while (continueSearch != LeafGroupArrayMap.STOP && currentLeaf != null) {
			if (continueSearch == LeafGroupArrayMap.CONTINUE_WITH_SCAN)
				continueSearch = currentLeaf.entries.continueScan(1, 0, highKey, results);
			else
				continueSearch = currentLeaf.entries.queryRange(lowKey, highKey, results);

			currentLeaf = currentLeaf.nextLeafGroup;
		}
		results.thatsallfolks();
	}

	/**
	 * Check whether the leaf group is empty or not
	 */
	public boolean isEmpty() {
		return entries.size() == 0;
	}
	
	/**
	 * Check whether a leaf in this leaf group is full or not.
	 * @param index - index of the leaf inside the leaf group i.e. (leaf number -1)
	 */
	public boolean isFull(int index){
		return entries.currentLeafSize[index] == (2*k_star);
	}

	/**
	 * Check whether CSBTreeNode implemention is leaf or not. 
	 */
	public boolean isLeaf() {
		return true;
	}

	public void toDot(OutputStream dest) {
	}
}
