package core.csbtree;

import java.io.OutputStream;

import util.IntPushOperator;

/**
 * Represents a node group from the csb-tree (either directory node group or leaf group).
 * 
 * @author alekh
 */
public interface CSBTreeNode {

	/**
	 * Indicates if this node is a leaf group.
	 * 
	 * @return
	 */
	public boolean isLeaf();

	/**
	 * Obtains all values mapped to the given key. Values are delivered to the
	 * provided push operator.
	 * 
	 * @param key
	 * @param results
	 */
	public void get(int index, int key, IntPushOperator results);

	/**
	 * Obtains all values mapped to the given key range (low and high,
	 * inclusive). Values are delivered to the provided push operator.
	 * 
	 * @param lowKey
	 * @param highKey
	 * @param results
	 */
	public void queryRange(int index, int lowKey, int highKey, IntPushOperator results);

	/**
	 * Adds the given mapping to the node.
	 * 
	 * @param key
	 * @param value
	 * @return
	 */
	public SplitInfo add(int index, int key, int value, boolean isParentNodeFull, int lowKey, int highKey, LeafCarrier leafCarrier);

	/**
	 * Removes a single instance of the key-value mapping from the node. If the
	 * value given is equal to BTreeConstants.ALL_MAPPINGS then all mappings
	 * associated to the given key will be removed.
	 * 
	 * @param key
	 * @param lowKey
	 * @param highKey
	 */
	public void remove(int index, int key, int value, int lowKey, int highKey);

	/**
	 * Indicates if a node/leaf is full.
	 * 
	 * @param index
	 * @return
	 */
	public boolean isFull(int index);
	
	/**
	 * Indicates if this node/leaf group is empty.
	 * 
	 * @return
	 */
	public boolean isEmpty();

	/**
	 * Serializes this node to a dotty representation (to be rendered with
	 * graphviz).
	 * 
	 * @param dest
	 */
	public void toDot(OutputStream dest);

	// do we need this method?
	// public boolean update(int key, int value);
}