package core.csbtree;

import java.io.IOException;
import java.io.OutputStream;

import util.IntPushOperator;


/**
 * A simple csbtree implementation. (key equal or less than pivot -> go right) Has
 * to be taylored to different key and value types manually as Generics would
 * use Complex type (inefficent) instead of native types. We have pointers among
 * leaves groups and also only store key/value mappings on the leaves. Therefore, this
 * is a CSB+-tree implementation.
 * <p>
 * The delete strategy implemented in this tree is to just remove and not merge.
 * Thus, there is no logic to merge at nodes at half occupation and nodes may
 * become underutilized. This may be monitored by calculating the utilization at
 * the leaf level.
 * 
 * @author alekh
 */
public class CSBTree {

	/** the root of the csb-tree */
	protected CSBTreeNode root = null;

	/** the left-most leaf on the csb-tree */
	protected LeafGroup firstLeaf = null;

	/** the degree of the csb-tree (internal nodes) */
	protected int k;

	/** the degree of the leaves */
	protected int k_star;

	private double leafUtilization;

	private int leafGroupCount;
	
	private int leafCount;

	private int elemCount;

	private boolean refreshNeeded = true;
	
	 private int addCounter = 0;
	 private int getCounter = 0;
	 private int delCounter = 0;
	 private int rangeQueryCounter = 0;
	

	/**
	 * Instantiates a new CBTree.
	 * 
	 * @param k
	 * @param k_star
	 */
	public CSBTree(int k, int k_star) {
		this.k = k;
		this.k_star = k_star;
	}

	/**
	 * Adds a mapping from key to value in the csb-tree. Duplicate mappings are
	 * allowed.
	 * 
	 * @param key
	 * @param value
	 * @return
	 */
	public void add(int key, int value, LeafCarrier leafCarrier) {
		if (root == null) {
			firstLeaf = new LeafGroup(k, k_star);
			root = firstLeaf;
		}
		
		
		
		SplitInfo splitInfo = root.add(0, key, value, false, Integer.MIN_VALUE, Integer.MAX_VALUE, leafCarrier);
		
		addCounter++;
		//System.out.println("numberOfAdds: " + addCounter);
		
		if (splitInfo != null) {
			// root overflow!:
			
			NodeGroup newRoot = new NodeGroup(splitInfo.leftNode, splitInfo.pivot, k);
			root = newRoot;
		}
		refreshNeeded = true;
	}

	public void add(int key, int value) {
		add(key, value, null);
	}

	/**
	 * Gets the value currently mapped to the given key.
	 * 
	 * @param key
	 * @return
	 */
	public void get(int key, IntPushOperator results) {		
		root.get(0, key, results);
		getCounter++;
		//System.out.println("Get Counter:"+getCounter);
	}

	/**
	 * Removes all mappings corresponding to the given key from the csb-tree.
	 * 
	 * @param key
	 * @return
	 */
	public void remove(int key) {
		root.remove(0, key, CSBTreeConstants.ALL_MAPPINGS, Integer.MIN_VALUE, Integer.MAX_VALUE);		
		refreshNeeded = true;
		delCounter++;
		//System.out.println("Del Counter:"+delCounter);
	}

	/**
	 * Removes one instance of the given key-value mapping from the csb-tree. Note
	 * that even if multiple instances of that mapping exist, only a single
	 * instance will be removed.
	 * 
	 * @param key
	 * @param value
	 */
	//public void remove(int key, int value) {
	//	root.remove(0, key, value, Integer.MIN_VALUE, Integer.MAX_VALUE);
	//	refreshNeeded = true;
	//}

	/**
	 * Returns all the values mapped in the given key range through the provided
	 * push operator. We include values that correspond to the lowKey and also
	 * include values that correspond to the highKey.
	 * 
	 * @param lowKey
	 * @param highKey
	 * @return
	 */
	public void queryRange(int lowKey, int highKey, IntPushOperator results) {
		root.queryRange(0, lowKey, highKey, results);
		rangeQueryCounter++;
	}

	/**
	 * Prints the root of the tree as a string.
	 */
	public String toString() {
		return root.toString();
	}

	/**
	 * generates a dotty representation of the tree in the given output stream.
	 */
	public void toDot(OutputStream dest) {
		try {
			// write header
			dest.write("digraph g {\n".getBytes());
			dest.write("node [shape=record,height=.1];\n".getBytes());

			// write tree internal nodes
			root.toDot(dest);

			// write tree leaf nodes
			LeafGroup currentLeaf = firstLeaf;
			while (currentLeaf != null) {
				currentLeaf.toDot(dest);
				currentLeaf = currentLeaf.nextLeafGroup;
			}

			// write trailer, flush, and close
			dest.write("}\n".getBytes());
			dest.flush();
			dest.close();
		} catch (IOException e) {
			System.out.println("could not write dotty");
			e.printStackTrace();
		}
	}

	public void printStats() {
		calculateStats();
		System.out.print("leafUtilization:\t" + leafUtilization + "\tleafCount:\t" + leafCount + "\telementCount:\t"
				+ elemCount + "\t");
	}

	public long size() {
		calculateStats();
		//System.out.println("No of leaf groups:"+leafGroupCount);
		//System.out.println("Additions: "+addCounter);
		//System.out.println("PointQueries: "+getCounter);
		//System.out.println("Deletions: "+delCounter);
		//System.out.println("RangeQueries: "+rangeQueryCounter);
		//int total = (addCounter+getCounter+delCounter+rangeQueryCounter);
		//System.out.println("Total: "+total);
		return elemCount;
	}

	private void calculateStats() {
		if (refreshNeeded) {
			LeafGroup currentLeaf = firstLeaf;
			elemCount = 0;
			leafCount = 0;
			leafGroupCount = 0;
			while (currentLeaf != null) {
				leafCount += currentLeaf.entries.leaves();
				elemCount += currentLeaf.entries.getKeyNumber();
				currentLeaf = currentLeaf.nextLeafGroup;
				leafGroupCount++;
			}

			leafUtilization = (double) elemCount / (double) (leafCount * 2 * k_star);
			refreshNeeded = false;
		}
	}

	public double getLeafUtilization() {
		calculateStats();
		return leafUtilization;
	}

	public int getLeafCount() {
		calculateStats();
		return leafCount;
	}

	public int getElemCount() {
		calculateStats();
		return elemCount;
	}

}
